% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tensorReg2D.R
\name{tensorReg2D}
\alias{tensorReg2D}
\title{Fitting Second-order Tensor Generalized Regression}
\usage{
tensorReg2D(y, X, W = NULL, n_R, family, opt = 1, max_ite = 100, tol = 10^(-7))
}
\arguments{
\item{y}{A numerical vector. Dependent variable.}

\item{X}{A numerical 3-D array Independent variable(3-D tensor).}

\item{W}{A numerical matrix. Independent variable.}

\item{n_R}{A numerical constant. A predefined value determines the rank of
the approximate matrix}

\item{family}{Family of \kbd{generalized linear model}. Provide three options for model.(see more details in
\strong{Details})}

\item{opt}{Optimization options. Provide two options for optimization
stopping criterion. \strong{opt = 1 or 2}. (see more details in
\strong{Details})}

\item{max_ite}{Maximum iteration. The value of maximum iterations for the
algorithm.}

\item{tol}{Tolerance. The value of tolerance with respect to optimization.}
}
\value{
tensorReg2D returns an object of \kbd{"tsglm"}.

 The function, \code{\link{summary.tsglm}} a customized method from generic function
 \code{\link[base]{summary}}, can be used to obtain and print a summary and analysis
 of variance table of the results.

 An object of class \kbd{tsglm} is a list containing at least the following components:

  \kbd{ite}: The number of executed times when stopping the function.

  \kbd{b_EST}: The estimated coefficients for numerical variables.

  \kbd{b_SD}: The estimated standard deviation for numerical variables.

  \kbd{b_PV}: The p-value for numerical variables.

  \kbd{B_EST}: The estimated coefficients for 3-D tensor variables.

  \kbd{B_SD}: The estimated standard deviation for 3-D tensor variables.

  \kbd{B_PV}: The p-value for 3-D tensor variables.

  \kbd{Residuals}: The differences between true values and prediction values. Provide for
  \kbd{family = "gaussian"}.

  \kbd{Dev_res}: Deviance residuals for glm. Provide for model except \kbd{family = "gaussian"}.

  \kbd{Dev}: The value of Null deviances and Residual deviance. Provide for model except
  \kbd{family = "gaussian"}.

  \kbd{IC}: The value of AIC and BIC.

  \kbd{DoF}: Degree of freedom.

  \kbd{call}: The formula of fitted model.

  \kbd{family}: The family for model.
}
\description{
\kbd{tensorReg2D} is used to fit second-order tensor generalized regression model. It mainly
focus on parameter estimation, including parameter coefficients and standard
deviation. The function is built upon \strong{Alternating Least Square
Algorithm}, so we provide two criterion to determine optimal result (see more
details below in Arguments). Also, we offer model complexity
measurement,including AIC and BIC.
}
\details{
\kbd{tensorReg2D} focuses on second-order tensor generalized regression problems.
To be more specific, it provides statistical inference for input variables.
Moreover, the function isn't restricted to second-order tensor input \kbd{X};
it could combine with other meaningful numerical variables \kbd{W}.

Since \kbd{tensorReg2D} is based on \strong{Alternating Least Square
Algorithm}, we need to pre-define following arguments to meet favorable
optimization result.

\kbd{n_R}: In the case of regression with the order 2, P-by-G-by-n tensor, we
can break a unknown parameter matrix \strong{B}(P-by-G) into multiplication
of two matrix \strong{B_1}(P-by-R) and \strong{t(B_2)} (R-by-G), which means
that we can estimate the original matrix \strong{B} by iteratively updating
\strong{B_1} and \strong{B_2}. In this scenario, \kbd{n_R} equals to the rank
of these two approximate matrix \strong{B_1} and \strong{B_2}. Conceivably,
\kbd{1 <= n_R <= min(P,G)}, and by properly pre-appointing \kbd{n_R}, we can
estimate a unknown parameter matrix. By default, \kbd{n_R = 1}.

\kbd{opt}: In optimization algorithm, we have to determine stopping
criterion. In \kbd{tensorReg2D}, we offer two criteria. \kbd{If opt = 1}, the
criterion is that we stop our execution when the maximum difference between
the elements among an estimated parameter matrix \strong{B} with an estimated
parameter vector \strong{b} and preceding ones is less than predefined
tolerance (\kbd{tol}) . \kbd{If opt = 2}, the criterion is that we stop our
execution when the maximum difference between the elements among an estimated
approximate parameter matrix \strong{B_1} , \strong{B_2} with an estimated
parameter vector \strong{b} and preceding ones is less than predefined
tolerance (\kbd{tol}).

\kbd{family}: In \kbd{tensorReg2D}, we provide three options for specific generalized regression
problem. First, \kbd{family = "gaussian"} using \kbd{identity} link function corresponds to linear regression
model, where dependent variable is real number. Next, \kbd{family = "binomial"} based on \kbd{logit} link function
corresponds to logistic regression, where dependent variable is restricted to zero or one binary variable. Finally,
\kbd{family = "poisson"} built upon \kbd{log} link function corresponds to poisson regression, where dependent
variable is non-negative integer.

\kbd{max_ite}: In optimization algorithm, we have to beforehand determine maximum iteration beforehand.
By default, \kbd{max_ite = 100}.

\kbd{tol}: In optimization algorithm, we have to beforehand determine maximum tolerance to cooperate with
stopping criterion(\kbd{opt}).
}
\examples{
# Simulation data
n <- 500 # number of observations
n_P <- 3; n_G <- 64 # dimension of 3-D tensor variables.
n_d <- 1 # number of numerical variable, if n_d == 1,  numerical variable equals to intercept.
beta_True <- rep(1, n_d)
B_True <- c(1,1,1)\%*\%t(rnorm(n_G)) + c(0, .5, .5)\%*\%t(rnorm(n_G))
B_True <- B_True / 10
W <- matrix(rnorm(n*n_d), n, n_d); W[,1] <- 1
X <- array(rnorm(n*n_P*n_G), dim=c(n_P, n_G, n))
## Regression
y_R<- as.vector(W\%*\%beta_True + X\%hp\%B_True + rnorm(n))
DATA_R <- list(y = y_R, X = X, W = W)
## Binomial
p_B <- exp(W\%*\%beta_True + X\%hp\%B_True); p_B <- p_B/(1+p_B)
y_B <- rbinom(n, 1, p_B)
DATA_B <- list(y = y_B, W = W, X = X)
## Poisson
p_P <- exp(W\%*\%beta_True + X\%hp\%B_True)
y_P <- rpois(n, p_P)
y_P[which(y_P > 170)] <- 170 # If y_P > 170, factorial(y_P) == inf.
DATA_P <- list(y = y_P, W = W, X = X)

# Execution
## Regression
result_R <- tensorReg2D(y = DATA_R$y, X = DATA_R$X, W=NULL, n_R = 1, family = "gaussian",
opt = 1, max_ite = 100, tol = 10^(-7) )
## Visualization
image(B_True);image(result_R$B_EST)
head(predict(result_R, DATA_R$X))

## Binomial
result_B <- tensorReg2D(y = DATA_B$y, X = DATA_B$X, W=NULL, n_R = 1, family = "binomial",
opt = 1, max_ite = 100, tol = 10^(-7) )
## Visualization
image(B_True);image(result_B$B_EST)
head(predict(result_B, DATA_B$X))

## Poisson
result_P <- tensorReg2D(y = DATA_P$y, X = DATA_P$X, W=NULL, n_R = 1, family = "poisson",
opt = 1, max_ite = 100, tol = 10^(-7) )
## Visualization
image(B_True);image(result_P$B_EST)
head(predict(result_P, DATA_P$X))

}
\references{
Mengyun Wu, Jian Huang, and Shuangge Ma (2017). Identifying gene-gene
  interactions using penalized tensor regression.

Sheng-Mao Chang, Meng Yang, Wenbin Lu, Yu-Jyun Huang, Yueyang Huang, Hung Hung,
Jeffrey C Miecznikowski, Tzu-Pin Lu, Jung-Ying Tzeng,
Gene-set integrative analysis of multi-omics data using tensor-based association test,
Bioinformatics, 2021;, btab125,
(\href{https://academic.oup.com/bioinformatics/advance-article-abstract/doi/10.1093/bioinformatics/btab125/6154849}{Link}))
}
\author{
Sheng-Mao Chang
}
