#' @title Compromise-admissible check
#' @description This function checks if the given game is compromise-admissible.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if the game is compromise-admissible, \code{FALSE} otherwise. If \code{instance=TRUE} and \eqn{\{i \in N : m_i(v)>M_i(v)\} \neq \emptyset}, one of the players in that set is also returned.
#' @details Let \eqn{v\in G^N}.
#' The utopia payoff of player \eqn{i\in N} is defined as \eqn{M_i(v)=v(N)-v(N\backslash i)}.
#' The minimal right of player \eqn{i\in N} is defined as \eqn{m_i(v)=\max_{S:i\in S}(v(S)-\sum_{j\in S\backslash i}M_j(v))}.
#'
#' The game \eqn{v\in G^N} is said to be compromise-admissible if its core-cover is not empty, that is, if the following conditions hold:
#'
#' 1) \eqn{m(v)\leq M(v)}.
#'
#' 2) \eqn{\sum_{i\in N}m_{i}(v)\leq v(N)\leq \sum_{i\in N}M_i(v)}.
#' @examples
#' compromiseadmissiblecheck(c(0,0,0,0,10,40,30,60,10,20,90,90,90,130,160))
#' compromiseadmissiblecheck(c(1,2,2), instance=TRUE)
#'
#' # What if the game is a cost game?
#' cost.v <- c(30, 20, 50, 40, 60, 60, 75) # compromise-admissible cost game
#' compromiseadmissiblecheck(-c(30, 20, 50, 40, 60, 60, 75))
#' @export

compromiseadmissiblecheck <- function(v, binary = FALSE, instance = FALSE) {

  ################################
  ### Comprobación datos entrada##
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables#####
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # Por defecto, no hay contraejemplo
  M <- c() # Utopia payoffs
  m <- c() # Minimal rights.
  tol <- 100*.Machine$double.eps

  ################################
  ###### Cuerpo de la función#####
  ################################

  # Utopia Payoffs:
  for (ii in 1:n) {
    M[ii] <- v[nC] - v[nC - 2^(ii - 1)] # Mi=v(N)-v(N\ i)
  }

  # Minimal rights:

  m <- v[2.^(seq(1:n) - 1)] # Vector de v(i).

  for (S in 1:nC) {
    num <- S # Primero calculo el vector en con los jugadores pertenecientes a la coalición en posición S.
    J <- floor(log2(num)) + 1
    Sjug <- rep(0, n) # Inicializo la coalición S

    for (ii in 1:J)
    {
      if (num %% 2 == 1) {
        Sjug[ii] <- 1
      }
      num <- num %/% 2
    }

    SMii <- M * Sjug # Pagos de utopia de los jugadores en la coalición S.
    for (kk in which(Sjug > 0)) { # max(v(S)-\sum_{j in S\i}M_j(N,v)))
      m[kk] <- max(m[kk], v[S] - sum(SMii) + SMii[kk])
    }
  }

  if (sum(M - m + tol < 0) > 0 || v[nC] - sum(m) + tol < 0 || sum(M) - v[nC] + tol < 0) { # Comprobamos si hay algún m mayor que algún M, si sum(m)>v(N) o si v(N)>sum(M).
    checkR <- FALSE
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (instance == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también el código numérico de la coalición S.
    exampleR <- NULL
    if (v[nC] - sum(m) + tol < 0) {
      message(paste("v(N) is greater than the sum of the utopia payoffs."))
    }
    if (sum(M) - v[nC] + tol < 0) {
      message(paste("v(N) is less than the sum of the minimal rights."))
    }
    if (sum(M - m + tol < 0) > 0) {
      exampleR <- min(which(M - m + tol < 0))
      if (binary==FALSE) {
        exampleR <- codebin2lex(n, exampleR)
      }
    }
    return(list(check = checkR, example = exampleR))
  }
} # Fin de la función
