THSVM<- function(data, h) {
  # Defensive input handling
  if (missing(data)) stop("`data` is required.")
  if (missing(h)) stop("`h` (forecast horizon) is required.")
  data <- as.numeric(data)
  h <- as.integer(h)
  if (h <= 0) stop("`h` must be a positive integer.")

  # Fit Theta model
  pp <- thetaf(data)    # requires forecast package
  kk <- as.numeric(pp$residuals)   # residuals (for SVM)
  kk1 <- as.numeric(pp$fitted)     # fitted values of Theta model
  p2 <- pp$model        # Theta coefficients / model info

  # Check stationarity / suitability (optional)
  w <- tryCatch(terasvirta.test(as.ts(kk)), error = function(e) NULL)
  p1 <- if (!is.null(w) && !is.null(w$p.value)) w$p.value else NA
  Test_Result <- if (is.na(p1)) {
    "terastvirta test failed or unavailable"
  } else if (p1 > 0.05) {
    "Data is not suitable for hybrid modelling"
  } else {
    "Data is suitable for hybrid modelling"
  }

  # Forecast from Theta
  ff <- forecast(pp, h = h)
  ff1 <- as.numeric(ff$mean)

  # Call ARSVM (user's function from TSSVM package); expect list elements
  zz <- ARSVM(kk, h) # keep inside tryCatch if you want to trap errors
  # Extract SVM outputs defensively
  zz1 <- if (!is.null(zz$`Model Summary`)) zz$`Model Summary` else zz$summary
  zz2 <- if (!is.null(zz$`Optimum lag`)) zz$`Optimum lag` else zz$optimum_lag
  fitv <- if (!is.null(zz$fitted)) as.numeric(zz$fitted) else as.numeric(zz$fit)
  ff2_raw <- if (!is.null(zz$forecasted.values)) as.numeric(zz$forecasted.values) else as.numeric(zz$forecast)

  # === Make ff2 a length-h vector ===
  ff2 <- NULL
  if (length(ff2_raw) == h) {
    ff2 <- ff2_raw
  } else if (length(ff2_raw) > 0 && (length(ff2_raw) %% h == 0)) {
    # If ff2_raw is concatenated forecasts (e.g., 2 rows x h columns flattened),
    # reshape into matrix with rows = length(ff2_raw) / h and take column means
    nr <- length(ff2_raw) / h
    mat <- matrix(ff2_raw, nrow = nr, byrow = TRUE)
    ff2 <- as.numeric(colMeans(mat, na.rm = TRUE))
    warning(sprintf("`zz$forecasted.values` had length %d; reshaped to %d x %d and averaged -> length %d",
                    length(ff2_raw), nr, h, length(ff2)))
  } else if (length(ff2_raw) < h) {
    # pad with last value to length h (safe fallback)
    ff2 <- c(ff2_raw, rep(tail(ff2_raw, 1), h - length(ff2_raw)))
    warning(sprintf("`zz$forecasted.values` shorter (%d) than h (%d) -> padded to length h", length(ff2_raw), h))
  } else {
    # longer but not divisible: truncate to h (keep first h)
    ff2 <- ff2_raw[1:h]
    warning(sprintf("`zz$forecasted.values` length %d not multiple of h; truncating to first %d elements", length(ff2_raw), h))
  }

  # === Align fitted values: fitv + pp2 ===
  # pp2 are the corresponding Theta fitted values aligned with fitv
  if (is.null(zz2) || !is.numeric(zz2)) {
    zz2 <- 0
    warning("`zz$`Optimum lag` not found or not numeric; assuming 0 (no lag).")
  }
  pp2_start <- zz2 + 1
  if (pp2_start <= 0) pp2_start <- 1
  if (pp2_start > length(kk1)) {
    stop("Computed start index for pp2 is beyond length of Theta fitted values.")
  }
  pp2 <- as.numeric(kk1[pp2_start:length(kk1)])
  fitv <- as.numeric(fitv)

  # Ensure fitv and pp2 have same length by truncating to the minimum
  n_fitted_common <- min(length(fitv), length(pp2))
  if (n_fitted_common <= 0) {
    stop("No overlapping fitted/residual series to compute hybrid fitted values.")
  }
  if (length(fitv) != length(pp2)) {
    warning(sprintf("`fitv` length %d and `pp2` length %d differ -> truncating both to %d", length(fitv), length(pp2), n_fitted_common))
  }
  fitv_trim <- fitv[seq_len(n_fitted_common)]
  pp2_trim <- pp2[seq_len(n_fitted_common)]
  pp3 <- fitv_trim + pp2_trim  # fitted values of hybrid model (aligned)

  # Ground truth segment for error calculation
  # original data values corresponding to pp3: data[(zz2+1):length(data)] trimmed to n_fitted_common
  data_segment_full <- as.numeric(data)
  if (pp2_start > length(data_segment_full)) {
    stop("Start index for computing errors is beyond data length.")
  }
  pp4_full <- data_segment_full[pp2_start:length(data_segment_full)]
  pp4 <- pp4_full[seq_len(n_fitted_common)]

  # Compute MAPE and MSE (safe: avoid division by zero)
  nonzero_idx <- which(pp4 != 0)
  if (length(nonzero_idx) == 0) {
    Mape <- NA
    warning("All actual values in pp4 are zero; MAPE not defined.")
  } else {
    Mape <- mean(abs((pp4[nonzero_idx] - pp3[nonzero_idx]) / pp4[nonzero_idx])) * 100
  }
  Mse <- mean((pp4 - pp3)^2)

  # Combine Theta forecast (ff1) and SVM forecast (ff2) carefully
  if (length(ff1) != length(ff2)) {
    minh <- min(length(ff1), length(ff2))
    warning(sprintf("Theta forecast length (%d) and SVM forecast length (%d) differ -> combining first %d elements", length(ff1), length(ff2), minh))
    ff1_use <- ff1[seq_len(minh)]
    ff2_use <- ff2[seq_len(minh)]
  } else {
    ff1_use <- ff1
    ff2_use <- ff2
  }
  ff4 <- ff1_use + ff2_use

  # Return a named list with helpful diagnostics
  out <- list(
    Test_Result = Test_Result,
    THETA_coefficients = p2,
    SVM_Summary = zz1,
    Optimum_lag = zz2,
    MAPE = Mape,
    MSE = Mse,
    fitted = pp3,
    forecasted.values = ff4
  )
  return(out)
}
