% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeEulerCharacteristic}
\alias{computeEulerCharacteristic}
\title{A Vector Summary of the Euler Characteristic Curve}

\description{
Vectorizes the Euler characteristic curve \deqn{\chi(t)=\sum_{k=0}^d (-1)^k\beta_k(t),} where \eqn{\beta_0,\beta_1,\ldots,\beta_d} are the Betti curves corresponding to persistence diagrams \eqn{D_0,D_1,\ldots,D_d} of dimensions \eqn{0,1,\ldots,d} respectively, all computed from the same filtration. The evaluation method depends on the argument \code{evaluate}.
}

\usage{
computeEulerCharacteristic(D, scaleSeq, maxhomDim = -1, evaluate = "intervals")
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}

\item{scaleSeq}{a numeric vector of increasing scale values used for vectorization.}

\item{maxhomDim}{the maximum homological dimension considered (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). If set to -1 (default), it uses the maximum dimension found in \code{D}.}

\item{evaluate}{a character string indicating the evaluation method. Must be either \code{"intervals"} (default) or \code{"points"}.}

}

\value{
A numeric vector containing elements computed using \code{scaleSeq}=\eqn{\{t_1,t_2,\ldots,t_n\}} according to the method specified by \code{evaluate}. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.

\itemize{
  \item \code{"intervals"}: Computes average values of the Euler characteristic curve over intervals defined by consecutive elements in \code{scaleSeq}:

  \deqn{\Big(\frac{1}{\Delta t_1}\int_{t_1}^{t_2}\chi(t)dt,\frac{1}{\Delta t_2}\int_{t_2}^{t_3}\chi(t)dt,\ldots,\frac{1}{\Delta t_{n-1}}\int_{t_{n-1}}^{t_n}\chi(t)dt\Big)\in\mathbb{R}^{n-1},}
where \eqn{\Delta t_k=t_{k+1}-t_k}.

  \item \code{"points"}: Computes values of the Euler characteristic curve at each point in \code{scaleSeq}:

  \deqn{(\chi(t_1),\chi(t_2),\ldots,\chi(t_n))\in\mathbb{R}^n.}
}
}

\author{Umar Islambekov}

\references{
1. Richardson, E., & Werman, M. (2014). Efficient classification using the Euler characteristic. Pattern Recognition Letters, 49, 99-106.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

scaleSeq = seq(0, 2, length.out = 11) # A sequence of scale values

# Compute a vector summary of the Euler characteristic curve
computeEulerCharacteristic(D, scaleSeq)
}
