#' @title Simulate competing risks endpoints data where censoring depends on missing covariates.
#' @description We are assuming three covariates, two continuous and one binary,
#' where we assume the first continous and the binary covariates have missing values, and the second continuous covariates is fully observed.
#' @param n Sample Size.
#' @param beta1 Regression Coefficients for the event of interest process (length of 3)
#' @param beta2 Regression Coefficients for the event of competing risk process (length of 3).
#' @param phi Regression Coefficients for the censoring process (length of 3).
#' @param gamma Regression Coefficients for the missingness (length of 3).
#' @param seed A random seed for data generation.
#'
#' @return A data frame with missing values.
#' @importFrom stats rnorm rbinom runif
#' @examples
#' # Simulate a data set with approximately 45% censoring, 35% cause 1,
#' # and 20% causes 2, and half of the subjects with at least one missing covariates.
#' generate.compt(n = 100, beta1 = c(1,1,-1), beta2 = c(2, 2, -2),
#' phi= c(-1,-1,-0.5), gamma = c(1,1,1,-1), seed = 112358)
#' @export

generate.compt <- function(n, beta1, beta2, phi, gamma, seed){
  set.seed(seed)

  x1 <- rnorm(n, mean = 0, sd = 1)
  x2 <- rbinom(n, 1, prob = 0.5)
  x3 <- rnorm(n, mean = 2, sd = 1)
  X <- matrix(c(x1,x2,x3), ncol = 3)

  lambda01 <- 1
  lambda02 <- 1

  lambda1 <- lambda01*exp(X%*%beta1)
  lambda2 <- lambda02*exp(X%*%beta2)

  p <- lambda1/(lambda1 + lambda2)
  cause <- rbinom(n, 1, 1 - p) + 1

  u <- runif(n, 0, 1)
  t <- -log(u)/(lambda1 + lambda2)

  v <- runif(n, 0, 1)
  c <- -log(v)/exp(X%*%phi)

  time = pmin(t, c)
  delta = ifelse(t <= c, cause, 0)
  df <- data.frame(time, delta, X, t, cause)


  Z <- matrix(c(rep(1, times = n), df$X3, df$cause-1, df$t), ncol = 4)
  mu <- exp(Z%*%gamma)/(1 + exp(Z%*%gamma))
  m <- rbinom(n, 1, mu)
  type <- sample(1:3, n, replace = TRUE)
  df$X1[m == 1 & type == 1] <- NA
  df$X1[m == 1 & type == 3] <- NA
  df$X2[m == 1 & type == 2] <- NA
  df$X2[m == 1 & type == 3] <- NA

  return(df[,-c(6,7)])
}
