#' @title Calculate Soil Water Content from dielectric constant
#' @description Calculate volumetric soil water content (SWC [L³/L³] from dielectric constant (DK) using different equations (e.g. Topp et al. 1980).
#' Contains mainly functions for peat and other organic soils.
#' @param DK dielectric constant
#' @param FUN character string specifying the polynominal function. See details.
#' @param bd bulk density [g/cm³] (needed for FUN == 'malicki_bd' and 'jacobsen_soil_prop')
#' @param ths porosity or saturated water content [L³/L³] (needed for 'malicki_ths')
#' @param clay content of clay [percent] (only needed for 'jacobsen')
#' @param SOM soil organic matter [percent] (for 'jacobsen_soil_prop')
#' @details
#' Possible functions (FUN) are': \cr
#' \describe{\item{'topp'}{Topp et al. (1980)}}
#' \describe{\item{'jacobsen'}{Jacobsen and Schjonning (1993), equation 2}}
#' \describe{\item{'jacobsen_soil_prop'}{Jacobsen and Schjonning (1993), equation 3 (bd, ths and SOM are required as input)}}
#' \describe{\item{'pepin_5cm'}{Pepin, S. et al. (1992), Table 2 Depth 5 cm}}
#' \describe{\item{'pepin'}{Pepin, S. et al. (1992), Table 2 Pooled data}}
#' \describe{\item{'roth_org'}{Roth et al. (1992), Tabel 3c}}
#' \describe{\item{'malicki_bd'}{Malicki, M.A. et al. (1996), equation 10 (bd is required as input)}}
#' \describe{\item{'malicki_ths'}{Malicki, M.A. et al. (1996), equation 12 (ths is required as input)}}
#' \describe{\item{'myllys'}{Myllys M. and Simojoki, A. (1996), Table 2 pooled}}
#' \describe{\item{'myllys_sphagnum'}{Myllys M. and Simojoki, A. (1996), Table 2 Sphagnum}}
#' \describe{\item{'myllys_carex'}{Myllys M. and Simojoki, A. (1996), Table 2 Carex}}
#' \describe{\item{'kellner'}{Kellner E., Lundin L.C. (2001), Table 2 Pooled data}}
#' \describe{\item{'kellner_h2'}{Kellner E., Lundin L.C. (2001), Table 2 H2}}
#' \describe{\item{'kellner_h3'}{Kellner E., Lundin L.C. (2001), Table 2 H3}}
#' \describe{\item{'kellner_h4'}{Kellner E., Lundin L.C. (2001), Table 2 H4}}
#' \describe{\item{'beckwith'}{Beckwith C.W. and Baird A.J. (2001), eq. 1}}
#' \describe{\item{'yoshikawa_deadmoss'}{from Tabel 1 in Nagare et al. (2001)}}
#' \describe{\item{'yoshikawa_livemoss'}{from Tabel 1 in Nagare et al. (2001)}}
#' \describe{\item{'nagare'}{Nagare et al. (2011), combined from Table 4}}
#' \describe{\item{'oleszczuk'}{from Table 1 in Oleszczuk et al. (2004)}}
#' \describe{\item{'gs3'}{Meter group, eq. 7}}
#' @return data.table with columns DK, SWC and FUN.
#' @references Beckwith C.W., Baird A.J. (2001): Effect of biogenic gas bubbles on water flow through poorly decomposed blanket peat. Water Resour. Res., 37(3), 551-558.
#' @references Decagon Device (2016): GS3 Water Content, EC an Temperature Sensors, Operators maual , Decagon Device, Inc 2365 NE Hopkins Court Pullman WA 99163.\cr
#' @references Jacobsen, O.H., Schjonning, P. (1993): A laboratory calibration of time domain reflectometry for soil water measurement including effects of bulk density and texture. Journal of Hydrology, 151(2-4), 147-157. \cr
#' @references Kellner E., Lundin L.C. (2001): Calibration of Time Domain Reflectometry for Water Content in Peat Soil, Uppsala University, Dept. of Earth Sciences/Hydrology,SE-752 36 Uppsala, Sweden. Hydrology Research, 32(4-5), 315-332.\cr
#' @references Malicki, M.A., Plagge, R., Roth, C.H. (1996): Improving the calibration of dielectric TDR soil moisture determination taking into account the solid soil. European Journal of Soil Science, 47:357-366.
#' @references Meter group: Operator manual GS3 (http://library.metergroup.com/Manuals/20429_GS3_Web.pdf)
#' @references Myllys M., Simojoki, A. (1996): Calibration of time domain reflectometry (TDR) for soil moisture measuremnts in cultivated peat soils, Agricultural Research centre of Finland, Institut of Cropand Soil Science, University of helsinki, Department of Applied Chemistry and Mircrobiology.\cr
#' @references Nagare, R.M., Schincariol, R. A., Quinton, W.L., Hayashi, M. (2011): Laboratory calibration of time domain reflectometry to determine moisture content in undisturbed peat samples. European journal of soil science, 62(4), 505-515. \cr
#' @references Oleszczuk R., Brandyk T., and Szatylowicz J., 1998: Possibilities of TDR method application to measure moisture content of peat-muck soil (in Polish). Zesz. Prob. Post. Nauk Roln., 458, 263-274. \cr
#' @references Oleszczuk, R., Brandyk, T., Gnatowski, T., & Szatylowicz, J. (2004). Calibration of TDR for moisture determination in peat deposits. International agrophysics, 18(2). \cr
#' @references Pepin, S., Plamondon, A.P., and Stein, J. (1992): Peat water-content measurement using time domain reflectrometry, Can. J. of Forest Res., Vol. 22, pp. 534-540\cr
#' @references Roth C.H., M.A. Malicki, R. Plagge (1992): Empirical evaluation of the relationship between soil dielectric constant and volumetric water content as the basis for calibration soil moisture measurements by TDR; Institute of Ecology, Technical University of Berlin, Berlin, Germany and *Polish Academy of sciences, Institute of Agrophysics, ul. Doswiadczalna 4, 20-236 Lublin, Poland; Journal of Soil Science, 1992, 43, 1-13.\cr
#' @references Topp, G.C., Davis, J.L., Annan, A.P. (1980): Electromagnetic determination of soil water content: Measurements in coaxial transmission lines. Water resources research, 16(3), 574-582. \cr
#' @references Yoshikawa, K., Overduin, P., Harden, J. (2004): Moisture content measurements of moss (Sphagnum spp.) using commercial sensors. Permafrost Periglac, 15, 309-318.\cr
#' @importFrom data.table data.table rbindlist
#' @export
DK_to_SWC <- function(DK,
                      FUN = c('topp', 'jacobsen', 'jacobsen_soil_prop', 'pepin_5cm', 'pepin', 'roth_org',
                                    'malicki_bd', 'malicki_ths', 'myllys', 'myllys_sphagnum', 'myllys_carex',
                                    'kellner', 'kellner_h2', 'kellner_h3', 'kellner_h4', 'beckwith','yoshikawa_deadmoss',
                                    'yoshikawa_livemoss', 'nagare', 'oleszczuk', 'gs3'),
                      bd, ths, clay, SOM) {
  FUN <- tolower(FUN)
  obj <- list()
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
  #  Calibrations on mineral soils
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
  # Topp
  if(any(FUN == 'topp')) {
    swc <- (-5.3*10^-2) + (2.92*10^-2*DK) - (5.5*10^-4 * DK^2) +	(4.3*10^-6*DK^3)
    obj$Topp <- data.table(DK = DK, SWC = swc, FUN = 'topp')
  }

  #Jacobsen
  if(any(FUN == 'jacobsen')) {
    swc <- (-7.01*10^-2) + (3.47*10^-2*DK) - (11.6*10^-4*DK^2) + (18*10^-6*DK^3)
    obj$jacobsen <- data.table(DK = DK, SWC = swc, FUN = 'jacobsen')
  }
  if(any(FUN == 'jacobsen_soil_prop')) {
    if(missing(bd)|missing(clay)|missing(SOM)){
      warning('bd, clay are SOM is needed for FUN == "jacobson_soil_prop"!')
    }
    if(!missing(bd)& !missing(clay) & !missing(SOM)){
   swc <- (-3.41* 10^-2) + (3.45*10^-2*DK) - (11.4*10^-4*DK^2) + (17.1*10^-6 * DK^3) -
      (3.7*10^-2 * bd) + (7.36*10^-4 * clay) + (47.7*10^-4* SOM)
    obj$jacobsen_soil_prop <- data.table(DK = DK, SWC = swc, FUN = 'jacobsen_soil_prop')
  }
  }
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  Calibrations on organic soils
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  ## Pepin (1991)
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  Sphagnum:
  #  From Table 1
  #  bulk density: 0.064
  #  particle density: 1.313
  #  Porosity: 95 %
  #  von Post: 1-2
  #  Ash content: 1.4
  #  pH of water suspension: 3.95
  if(any(FUN =='pepin_5cm')) {
    swc <- (-0.03) + (0.0248*DK) - (1.48*10^-4*DK^2)
    obj$pepin_5cm <- data.table(DK = DK, SWC = swc, FUN = 'pepin_5cm')
  }
  # ToDo: Pepin 10, 20, 30, 40, 50 cm

  # Pooled with data of 6 depth (5, 10, 20, 30, 40, 50 cm)
  # forest bog, ground surface mostly covered by Sphagnum
  if(any(FUN =='pepin')) {
    swc <- (0.085) + (0.0192*DK) - (0.95*10^-4*DK^2)
    obj$pepin <- data.table(DK = DK, SWC = swc, FUN = 'pepin')
  }
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  Roth et al. 1992
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  Fibric Histosol (two samples; 0-20 cm and 50-70 cm depth) and Cambic aerenosol (5 samples; 8-6, 65, 5-3, 3-0, 0-10 cm depth)
  #  bulk densities 0.26 to 0.77 [g cm3]
  #  porosities 0.527 - 0.786
  #  organic matter 10.5 to 54.8
  if(any(FUN =='roth_org')){
    swc <- (-0.0233) + (0.0285*DK ) + (-4.31*10^-4*DK^2) + (3.04*10^-6*DK^3)
    obj$roth_org <- data.table(DK = DK, SWC = swc, FUN = 'roth_org')
  }
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #   Malicki (1996)
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #
  # Large dataset (n = 62) of different soils containing
  # mineral (n = 21), organic (n = 13), different mixtures
  # (e.g. Peat-loess mixtures or Peat-sand mixtures, n = 12) and other capillary-porous materials
  # (e.g. river sand, pine bark; n = 15)

  ## Peat soils:
  #  Soil types (FAO): Arenic Cambisol, Terric Histosol, Fibric Histosol, Cambic Arenosol
  #  bulk densities: 0.13 - 0.58 [g/cm3]
  #  particel densities: 0.7 - 1.95 [g/cm3]
  #  SOC: 105 - 548 [g/kg]
  #
  # Account for influence of matrix on dielectric moisture readings by bulk density
  if(any(FUN =='malicki_bd')){
    if(missing(bd)){
      warning('bd is needed for FUN == "malicki_bd"!')
    }
    if(!missing(bd)){
      swc <- ((sqrt(DK)-0.819-0.168*bd - 0.159*bd^2)/ (7.17+1.18*bd))
      obj$malicki_bd <- data.table(DK = DK, SWC = swc, FUN = 'malicki_bd')
    }
  }
  # Account for influence of matrix on dielectric moisture readings by porosity (ths)
  if(any(FUN =='malicki_ths')){
    if(missing(ths)){
      warning('ths is needed for FUN == "malicki_ths"!')
    }
    if(!missing(ths)){
    swc <- ((sqrt(DK)- 3.47 + (6.22*ths) - (3.82*ths^2))/
                          (7.01 + 6.89*ths - 7.83*ths^2))
    obj$malicki_ths <- data.table(DK = DK, SWC = swc, FUN = 'malicki_ths')
    }
  }
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  Myllys and Simojoki (1996)
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  ## Peat Properties of samples used for calibration (Table 1):
  ## Sphagnum:
  #  Bulk density: 0.4
  #  Porosity: 0.77 [cm3/cm3]
  #  ash content 48 %DM
  #  pH 5.1
  ## Carex:
  #  Bulk density: 0.3
  #  Porosity: 0.75 [cm3/cm3]
  #  ash content 37 %DM
  #  pH 5.0
  #pooled Data
  if(any(FUN =='myllys')){
    swc <- (-0.0733) + (0.0417*DK ) + (-8.01*10^-4*DK^2) + (5.56*10^-6*DK^3)
    obj$myllys <- data.table(DK = DK, SWC = swc, FUN = 'myllys')
  }
  #sphagnum
  if(any(FUN =='myllys_sphagnum')){
    swc <- (-0.0830) + (0.0432*DK ) + (-8.53*10^-4*DK^2) + (5.99*10^-6*DK^3)
    obj$myllys_sphagnum <- data.table(DK = DK, SWC = swc, FUN = 'myllys_sphagnum')
  }
  # Carex
  if(any(FUN =='myllys_carex')){
    swc <- (-0.0986) + (0.0428*DK ) + (-8.08*10^-4*DK^2) + (5.52*10^-6*DK^3)
    obj$myllys_carex <- data.table(DK = DK, SWC = swc, FUN = 'myllys_carex')
  }

  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  Kellner und Lundin (2001)
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  ## undisturbed bog samples (Sweden)
  #  Bog hollows:
  #  - Sphagnum Species of the Cuspidata section (S. balticum, S. tenellum)
  #  - Eriophorum vaginatum
  #  Ridges and hummocks:
  #  - denser Sphagnum species (S.fuscum, S. rubellum)
  #  - caluna vulgaris and Empetrum nigrum
  ## Soil properties:
  #  von Post: H2, H3, H4
  #  bulk density: 0.0291 - 0.0662 [g/cm3]
  #  porosity: 0.953 - 0.979 [cm3/cm3]

  # Parameter in Table 2 of reference
  #  Pooled data (n = 190); error of SWC estimate 0.04, r-squared 0.968
  if(any(FUN == 'kellner')) { # calibrated on Sphagnum
    swc <- (3.90*10^-2) + (3.17*10^-2*DK)	- (4.50*10^-4*DK^2) + (2.60*10^-6*DK^3)
    obj$kellner <- data.table(DK = DK, SWC = swc, FUN = 'kellner')
  }

  # von Post H2 (n = 56); error of SWC estimate 0.03, r-squared 0.989
  if(any(FUN =='kellner_h2')){
    swc <- (-0.061) + (0.029*DK ) + (-4.5*10^-4*DK^2) + (3.0*10^-6*DK^3)
    obj$kellner_h2 <- data.table(DK = DK, SWC = swc, FUN = 'kellner_h2')
  }

  # von Post: H3 (n = 98); error of SWC estimate 0.04, r-squared 0.976
  if(any(FUN =='kellner_h3')){
    swc <- (0.059) + (0.028*DK ) + (-3.2*10^-4*DK^2) + (1.4*10^-6*DK^3)
    obj$kellner_h3 <- data.table(DK = DK, SWC = swc, FUN = 'kellner_h3')
  }

  # von Post: H4 (n = 36); error of SWC estimate 0.04, r-squared 0.948
  if(any(FUN =='kellner_h4')){
    swc <- (-0.044) + (0.034*DK ) + (-5.1*10^-4*DK^2) + (3.0*10^-6*DK^3)
    obj$kellner_h4 <- data.table(DK = DK, SWC = swc, FUN = 'kellner_h4')
  }

## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  Beckwith and Baird (2001)
## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  Calibration based on two samples
#  DK range: 29 - 75
## Peat:
#  mainly Sphagnum cuspidatum and some stems and roots of Eriophorum angustifolium
#  von Post: H2-H1 (upper), H3 lower
#  Porosity: 0.96 (lower) - 0.97 (upper) [cm3/cm3]

if(any(FUN =='beckwith')){
  if (any(DK < 29) | any(DK > 75)) {
    warning('FUN == "beckwith": DK values out of calibration range!')
  }
  swc <- (-0.40289) + (0.064591*DK ) + (-12.06*10^-4*DK^2) + (7.92*10^-6*DK^3)
  obj$beckwith <- data.table(DK = DK, SWC = swc, FUN = 'beckwith')
}

## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  Yoshikawa et al. 2004
## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
## Yoshikawa_deadmoss:
#  - dead sphagnum moss
#  - Equation is taken from Nagare et al. 2011
if(any(FUN =='yoshikawa_deadmoss')){
  swc <- (-0.6286) + (0.4337*sqrt(DK)) + (-5.49*10^-2*sqrt(DK)^2) + (0.33*10^-2*sqrt(DK)^3)
  obj$yoshikawa_deadmoss <- data.table(DK = DK, SWC = swc, FUN = 'yoshikawa_deadmoss')
}
## Yoshikawa_livemoss:
#  - live Sphagnum moss
#  - equation is taken from Nagare et al. 2011
if(any(FUN == 'yoshikawa_livemoss')){
  swc <- (-0.16625) + (0.1108*sqrt(DK)) + (-0.21*10^-2*sqrt(DK)^2) + (4.33*10^-4*sqrt(DK)^3)
  obj$yoshikawa_livemoss <- data.table(DK = DK, SWC = swc, FUN = 'yoshikawa_livemoss')
}
## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  Nagare et al. (2011)
## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  ToDo: add calibrations of all soils (table 4; nagare_m1,...) and Soil properties (table 3)
#  From table 4: Combined from 6 samples from two locations
#  L: high on peat plateau (Labrador tea and lichen cover) (2 samples)
#  M: low level at peat plateau-bog (Sphagnum moss dominant vegetation) (4 Samples)
#  porosities 73.06 - 93.73
#  bulk densities 0.0534 - 0.1831
if(any(FUN == 'nagare')) {
  swc <- (-1.89*10^-2) + (3.20*10^-2*DK)	- (4.59*10^-4*DK^2) + (2.70*10^-6*DK^3)
  obj$nagare <- data.table(DK = DK, SWC = swc, FUN = 'nagare')
}

## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  Oleszczuk et al. (1998) (in Polish)
## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#  Alder peat
if(any(FUN =='oleszczuk')){
  swc <- (-0.00276) + (0.02477*DK ) + (-3.15*10^-4*DK^2) + (0.02*10^-4*DK^3)
  obj$oleszczuk <- data.table(DK = DK, SWC = swc, FUN = 'oleszczuk')
}
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #  GS3 Decagon
  ## ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  ## organic soil
  #  Probe calibration from decagon
  if(any(FUN =='gs3')){
   swc <- 0.118*sqrt(DK) -0.117
   obj$gs3 <- data.table(DK = DK, SWC = swc, FUN = 'gs3')
  }
  obj <- rbindlist(obj)
  obj[]
}
