% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SampleSize.R
\name{sampleSize}
\alias{sampleSize}
\title{Sample Size Calculation for Bioequivalence and Multi-Endpoint Studies}
\usage{
sampleSize(
  mu_list,
  varcov_list = NA,
  sigma_list = NA,
  cor_mat = NA,
  sigmaB = NA,
  Eper,
  Eco,
  rho = 0,
  TAR = rep(1, length(mu_list)),
  arm_names = NA,
  ynames_list = NA,
  type_y = NA,
  list_comparator = NA,
  list_y_comparator = NA,
  power = 0.8,
  alpha = 0.05,
  lequi.tol = NA,
  uequi.tol = NA,
  list_lequi.tol = NA,
  list_uequi.tol = NA,
  dtype = "parallel",
  ctype = "ROM",
  vareq = TRUE,
  lognorm = TRUE,
  k = NA,
  adjust = "no",
  dropout = NA,
  nsim = 5000,
  seed = 1234,
  ncores = 1,
  optimization_method = "fast",
  lower = 2,
  upper = 500,
  step.power = 6,
  step.up = TRUE,
  pos.side = FALSE,
  maxiter = 1000,
  verbose = FALSE
)
}
\arguments{
\item{mu_list}{Named list of arithmetic means per treatment arm. Each element is a vector representing expected outcomes for all endpoints in that arm.}

\item{varcov_list}{List of variance-covariance matrices, where each element corresponds to a comparator. Each matrix has dimensions: number of endpoints × number of endpoints.}

\item{sigma_list}{List of standard deviation vectors, where each element corresponds to a comparator and contains one standard deviation per endpoint.}

\item{cor_mat}{Matrix specifying the correlation structure between endpoints, used along with \code{sigma_list} to calculate \code{varcov_list} if not provided.}

\item{sigmaB}{Numeric. Between-subject variance for a 2×2 crossover design.}

\item{Eper}{Optional numeric vector of length 2 specifying the period effect in a \code{dtype = "2x2"} design, applied as \code{c(Period 0, Period 1)}. Defaults to \code{c(0, 0)}. Ignored for \code{dtype = "parallel"}.}

\item{Eco}{Optional numeric vector of length 2 specifying the carry-over effect per arm in a \code{dtype = "2x2"} design, applied as \code{c(Reference, Treatment)}. Defaults to \code{c(0, 0)}. Ignored for \code{dtype = "parallel"}.}

\item{rho}{Numeric. Correlation parameter applied uniformly across all endpoint pairs. Used with \code{sigma_list} to compute \code{varcov_list} when \code{cor_mat} or \code{varcov_list} are not provided.}

\item{TAR}{Numeric vector specifying treatment allocation rates per arm. The order must match \code{arm_names}. Defaults to equal allocation across arms if not provided.}

\item{arm_names}{Optional character vector of treatment names. If not supplied, names are derived from \code{mu_list}.}

\item{ynames_list}{Optional list of vectors specifying endpoint names per arm. If names are missing, arbitrary names are assigned based on order.}

\item{type_y}{Integer vector indicating endpoint types: \code{1} for co-primary endpoints, \code{2} for secondary endpoints.}

\item{list_comparator}{List of comparators. Each element is a vector of length 2 specifying the treatment names being compared.}

\item{list_y_comparator}{List of endpoint sets per comparator. Each element is a vector containing endpoint names to compare. If not provided, all endpoints common to both comparator arms are used.}

\item{power}{Numeric. Target power (default = 0.8).}

\item{alpha}{Numeric. Significance level (default = 0.05).}

\item{lequi.tol}{Numeric. Lower equivalence bounds (e.g., -0.5) applied uniformly across all endpoints and comparators.}

\item{uequi.tol}{Numeric. Upper equivalence bounds (e.g., 0.5) applied uniformly across all endpoints and comparators.}

\item{list_lequi.tol}{List of numeric vectors specifying lower equivalence bounds per comparator.}

\item{list_uequi.tol}{List of numeric vectors specifying upper equivalence bounds per comparator.}

\item{dtype}{Character. Trial design: \code{"parallel"} (default) for parallel-group or \code{"2x2"} for crossover (only for 2-arm studies).}

\item{ctype}{Character. Hypothesis test type: \code{"DOM"} (Difference of Means) or \code{"ROM"} (Ratio of Means).}

\item{vareq}{Logical. Assumes equal variances across arms if \code{TRUE} (default = \code{FALSE}).}

\item{lognorm}{Logical. Whether data follows a log-normal distribution (\code{TRUE} or \code{FALSE}).}

\item{k}{Integer vector. Minimum number of successful endpoints required for global bioequivalence per comparator. Defaults to all endpoints per comparator.}

\item{adjust}{Character. Alpha adjustment method: \code{"k"} (K-fold), \code{"bon"} (Bonferroni), \code{"sid"} (Sidak), \code{"no"} (default, no adjustment), or \code{"seq"} (sequential).}

\item{dropout}{Numeric vector specifying dropout proportion per arm.}

\item{nsim}{Integer. Number of simulated studies (default = 5000).}

\item{seed}{Integer. Seed for reproducibility.}

\item{ncores}{Integer. Number of processing cores for parallel computation. Defaults to \code{1}. Set to \code{NA} for automatic detection (\code{ncores - 1}).}

\item{optimization_method}{Character. Sample size optimization method: \code{"fast"} (default, root-finding algorithm) or \code{"step-by-step"}.}

\item{lower}{Integer. Minimum sample size for search range (default = 2).}

\item{upper}{Integer. Maximum sample size for search range (default = 500).}

\item{step.power}{Numeric. Initial step size for sample size search, defined as \code{2^step.power}. Used when \code{optimization_method = "fast"}.}

\item{step.up}{Logical. If \code{TRUE} (default), search increments upward from \code{lower}; if \code{FALSE}, decrements downward from \code{upper}. Used when \code{optimization_method = "fast"}.}

\item{pos.side}{Logical. If \code{TRUE}, finds the smallest integer \code{i} closest to the root such that \code{f(i) > 0}. Used when \code{optimization_method = "fast"}.}

\item{maxiter}{Integer. Maximum iterations allowed for sample size estimation (default = 1000). Used when \code{optimization_method = "fast"}.}

\item{verbose}{Logical. If \code{TRUE}, prints progress and messages during execution (default = \code{FALSE}).}
}
\value{
A list containing:
\describe{
\item{\code{response}}{Array summarizing simulation results, including estimated sample sizes, achieved power, and confidence intervals.}
\item{\code{table.iter}}{Data frame showing estimated sample sizes and calculated power at each iteration.}
\item{\code{table.test}}{Data frame containing test results for all simulated trials.}
\item{\code{param.u}}{Original input parameters.}
\item{\code{param}}{Final adjusted parameters used in sample size calculation.}
\item{\code{param.d}}{Trial design parameters used in the simulation.}
}
}
\description{
Computes the required sample size to achieve a target power in studies with multiple endpoints and treatment arms.
The function employs modified root-finding algorithms to estimate sample size while accounting for correlation structures, variance assumptions,
and equivalence bounds across endpoints. It is particularly useful for bioequivalence trials and multi-arm studies with complex endpoint structures.
}
\examples{
mu_list <- list(SB2 = c(AUCinf = 38703, AUClast = 36862, Cmax = 127.0),
                EUREF = c(AUCinf = 39360, AUClast = 37022, Cmax = 126.2),
                USREF = c(AUCinf = 39270, AUClast = 37368, Cmax = 129.2))

sigma_list <- list(SB2 = c(AUCinf = 11114, AUClast = 9133, Cmax = 16.9),
                   EUREF = c(AUCinf = 12332, AUClast = 9398, Cmax = 17.9),
                   USREF = c(AUCinf = 10064, AUClast = 8332, Cmax = 18.8))

# Equivalent boundaries
lequi.tol <- c(AUCinf = 0.8, AUClast = 0.8, Cmax = 0.8)
uequi.tol <- c(AUCinf = 1.25, AUClast = 1.25, Cmax = 1.25)

# Arms to be compared
list_comparator <- list(EMA = c("SB2", "EUREF"),
                        FDA = c("SB2", "USREF"))

# Endpoints to be compared
list_y_comparator <- list(EMA = c("AUCinf", "Cmax"),
                          FDA = c("AUClast", "Cmax"))

# Equivalence boundaries for each comparison
lequi_lower <- c(AUCinf = 0.80, AUClast = 0.80, Cmax = 0.80)
lequi_upper <- c(AUCinf = 1.25, AUClast = 1.25, Cmax = 1.25)

# Run the simulation
sampleSize(power = 0.9, alpha = 0.05, mu_list = mu_list,
           sigma_list = sigma_list, list_comparator = list_comparator,
           list_y_comparator = list_y_comparator,
           list_lequi.tol = list("EMA" = lequi_lower, "FDA" = lequi_lower),
           list_uequi.tol = list("EMA" = lequi_upper, "FDA" = lequi_upper),
           adjust = "no", dtype = "parallel", ctype = "ROM", vareq = FALSE,
           lognorm = TRUE, ncores = 1, nsim = 50, seed = 1234)
}
\references{
Schuirmann, D. J. (1987). A comparison of the Two One-Sided Tests procedure and the
Power approach for assessing the equivalence of average bioavailability.
\emph{Journal of Pharmacokinetics and Biopharmaceutics, 15}(6), 657-680.
\url{doi:10.1007/BF01068419}

Mielke, J., Jones, B., Jilma, B., & König, F. (2018). Sample size for multiple hypothesis
testing in biosimilar development. \emph{Statistics in Biopharmaceutical Research, 10}(1), 39-49.
\url{doi:10.1080/19466315.2017.1371071}

Berger, R. L., & Hsu, J. C. (1996). Bioequivalence trials, intersection-union tests, and
equivalence confidence sets. \emph{Statistical Science}, 283-302.

Sozu, T., Sugimoto, T., Hamasaki, T., & Evans, S. R. (2015). "Sample Size Determination in
Clinical Trials with Multiple Endpoints." \emph{SpringerBriefs in Statistics}.
\url{doi:10.1007/978-3-319-22005-5}
}
\author{
Johanna Muñoz \email{johanna.munoz@fromdatatowisdom.com}
}
