% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PMCMC.R
\name{PMCMC}
\alias{PMCMC}
\alias{PMCMC.PMCMC}
\alias{PMCMC.default}
\title{Runs particle MCMC algorithm}
\usage{
PMCMC(x, ...)

\method{PMCMC}{PMCMC}(
  x,
  niter = 1000,
  nprintsum = 100,
  adapt = TRUE,
  adaptmixprop = 0.05,
  nupdate = 100,
  ...
)

\method{PMCMC}{default}(
  x,
  priors,
  func,
  u,
  npart = 100,
  iniPars = NA,
  fixpars = FALSE,
  niter = 1000,
  nprintsum = 100,
  adapt = TRUE,
  propVar = NA,
  adaptmixprop = 0.05,
  nupdate = 100,
  ...
)
}
\arguments{
\item{x}{A \code{PMCMC} object, or a \code{data.frame} containing time series count data, 
with the first column called \code{t}, followed by columns of time-series counts. 
The time-series counts columns must be in the order of the `counts` object in the
`func` function (see below).}

\item{...}{Not used here.}

\item{niter}{An integer specifying the number of iterations to run the MCMC.}

\item{nprintsum}{Prints summary of MCMC to screen every \code{nprintsum} iterations. 
Also defines how often adaptive scaling of proposal variances occur.}

\item{adapt}{Logical determining whether to use adaptive proposal or not.}

\item{adaptmixprop}{Mixing proportion for adaptive proposal.}

\item{nupdate}{Controls when to start adaptive update.}

\item{priors}{A \code{data.frame} containing columns \code{parnames}, \code{dist}, \code{p1} and 
\code{p2}, with number of rows equal to the number of parameters. The column
\code{parname} simply gives names to each parameter for plotting and summarising.
Each entry in the \code{dist} column must contain one of \code{c("unif", "norm", "gamma")}, 
and the corresponding \code{p1} and \code{p2} entries relate to the hyperparameters 
(lower and upper bounds in the uniform case; mean and standard deviation in the 
normal case; and shape and rate in the gamma case).}

\item{func}{A \code{SimBIID_model} object or an \code{XPtr} to simulation function. If the latter, 
then this function must take the following arguments in order: 
\itemize{
\item \code{NumericVector pars}: a vector of parameters;
\item \code{double tstart}: the start time;
\item \code{double tstop}: the end time;
\item \code{IntegerVector u}: a vector of states at time \code{tstart};
\item \code{IntegerVector counts}: a vector of observed counts at \code{tstop}.}}

\item{u}{A named vector of initial states.}

\item{npart}{An integer specifying the number of particles for the bootstrap particle filter.}

\item{iniPars}{A named vector of initial values for the parameters of the model. If left unspecified, 
then these are sampled from the prior distribution(s).}

\item{fixpars}{A logical determining whether to fix the input parameters (useful for 
determining the variance of the marginal likelihood estimates).}

\item{propVar}{A numeric (npars x npars) matrix with log (or logistic) covariances to use
as (initial) proposal matrix. If left unspecified then defaults to 
\code{diag(nrow(priors)) * (0.1 ^ 2) / nrow(priors)}.}
}
\value{
If the code throws an error, then it returns a missing value (\code{NA}). If 
        \code{fixpars = TRUE} it returns a list of length 2 containing:
\item{\code{output}:}{ a matrix with two columns. The first contains the simulated
         log-likelihood, and the second is a binary indicator relating to whether the
         simulation was 'skipped' or not (1 = skipped, 0 = not skipped);}
\item{\code{pars}:}{ a vector of parameters used for the simulations.}

If \code{fixpars = FALSE}, the routine returns a \code{PMCMC} object, essentially a 
         \code{list} containing:
 \item{\code{pars}:}{ an \code{mcmc} object containing posterior samples for the parameters;}
 \item{\code{u}:}{ a copy of the \code{u} input;}
 \item{\code{accrate}:}{ the cumulative acceptance rate;}
 \item{\code{npart}:}{ the chosen number of particles;}
 \item{\code{time}:}{ the time taken to run the routine (in seconds);}
 \item{\code{propVar}:}{ the proposal covariance for the parameter updates;}
 \item{\code{data}:}{ a copy of the \code{x} input;}
 \item{\code{priors}:}{ a copy of the \code{priors} input;}
 \item{\code{func}:}{ a copy of the \code{func} input.}
}
\description{
Runs particle Markov chain Monte Carlo (PMCMC) algorithm using 
     a bootstrap particle filter for fitting infectious disease models to 
     time series count data.
}
\details{
Function runs a particle MCMC algorithm using a bootstrap particle filter for a given model. 
                     If running with \code{fixpars = TRUE} then this runs \code{niter} simulations
                     using fixed parameter values. This can be used to optimise the number of 
                     particles after a training run. Also has \code{print()}, \code{summary()},
                     \code{plot()}, \code{predict()} and \code{window()} methods.
}
\examples{
\donttest{
## set up data to pass to PMCMC
flu_dat <- data.frame(
    t = 1:14,
    Robs = c(3, 8, 26, 76, 225, 298, 258, 233, 189, 128, 68, 29, 14, 4)
)

## set up observation process
obs <- data.frame(
    dataNames = "Robs",
    dist = "pois",
    p1 = "R + 1e-5",
    p2 = NA,
    stringsAsFactors = FALSE
)

## set up model (no need to specify tspan
## argument as it is set in PMCMC())
transitions <- c(
    "S -> beta * S * I / (S + I + R + R1) -> I", 
    "I -> gamma * I -> R",
    "R -> gamma1 * R -> R1"
)
compartments <- c("S", "I", "R", "R1")
pars <- c("beta", "gamma", "gamma1")
model <- mparseRcpp(
    transitions = transitions, 
    compartments = compartments,
    pars = pars,
    obsProcess = obs
)

## set priors
priors <- data.frame(
    parnames = c("beta", "gamma", "gamma1"), 
    dist = rep("unif", 3), 
    stringsAsFactors = FALSE)
priors$p1 <- c(0, 0, 0)
priors$p2 <- c(5, 5, 5)

## define initial states
iniStates <- c(S = 762, I = 1, R = 0, R1 = 0)

set.seed(50)

## run PMCMC algorithm
post <- PMCMC(
    x = flu_dat, 
    priors = priors,
    func = model, 
    u = iniStates,
    npart = 25,
    niter = 5000, 
    nprintsum = 1000
)

## plot MCMC traces
plot(post, "trace")

## continue for some more iterations
post <- PMCMC(post, niter = 5000, nprintsum = 1000)

## plot traces and posteriors
plot(post, "trace")
plot(post)

## remove burn-in
post <- window(post, start = 5000)

## summarise posteriors
summary(post)
}

}
\references{
Andrieu C, Doucet A and Holenstein R (2010) <doi:10.1111/j.1467-9868.2009.00736.x>
}
\seealso{
\code{\link{print.PMCMC}}, \code{\link{plot.PMCMC}}, \code{\link{predict.PMCMC}}, \code{\link{summary.PMCMC}}
    \code{\link{window.PMCMC}}
}
