% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/source_compete.R
\name{bqr}
\alias{bqr}
\title{Bayesian quantile regression}
\usage{
bqr(
  y,
  X,
  tau = 0.5,
  X_test = X,
  psi = length(y),
  nsave = 1000,
  nburn = 1000,
  nskip = 0,
  verbose = TRUE
)
}
\arguments{
\item{y}{\code{n x 1} vector of observed counts}

\item{X}{\code{n x p} matrix of predictors (no intercept)}

\item{tau}{the target quantile (between zero and one)}

\item{X_test}{\code{n_test x p} matrix of predictors for test data;
default is the observed covariates \code{X}}

\item{psi}{prior variance (g-prior)}

\item{nsave}{number of MCMC iterations to save}

\item{nburn}{number of MCMC iterations to discard}

\item{nskip}{number of MCMC iterations to skip between saving iterations,
i.e., save every (nskip + 1)th draw}

\item{verbose}{logical; if TRUE, print time remaining}
}
\value{
a list with the following elements:
\itemize{
\item \code{coefficients} the posterior mean of the regression coefficients
\item \code{fitted.values} the estimated \code{tau}th quantile at test points \code{X_test}
\item \code{post_theta}: \code{nsave x p} samples from the posterior distribution
of the regression coefficients
\item \code{post_ypred}: \code{nsave x n_test} samples
from the posterior predictive distribution at test points \code{X_test}
\item \code{post_qtau}: \code{nsave x n_test} samples of the \code{tau}th conditional quantile at test points \code{X_test}
\item \code{model}: the model fit (here, \code{bqr})
}
as well as the arguments passed
}
\description{
MCMC sampling for Bayesian quantile regression.
An asymmetric Laplace distribution is assumed for the errors,
so the regression models targets the specified quantile.
A g-prior is assumed for the regression coefficients.
}
\note{
The asymmetric Laplace distribution is advantageous because
it links the regression model (\code{X\%*\%theta}) to a pre-specified
quantile (\code{tau}). However, it is often a poor model for
observed data, and the semiparametric version \code{\link{sbqr}} is
recommended in general.

An intercept is automatically added to \code{X} and
\code{X_test}. The coefficients reported do *not* include
this intercept parameter.
}
\examples{
# Simulate some heteroskedastic data (no transformation):
dat = simulate_tlm(n = 100, p = 5, g_type = 'box-cox', heterosked = TRUE, lambda = 1)
y = dat$y; X = dat$X # training data
y_test = dat$y_test; X_test = dat$X_test # testing data

# Target this quantile:
tau = 0.05

# Fit the Bayesian quantile regression model:
fit = bqr(y = y, X = X, tau = tau, X_test = X_test)
names(fit) # what is returned

# Posterior predictive checks on testing data: empirical CDF
y0 = sort(unique(y_test))
plot(y0, y0, type='n', ylim = c(0,1),
     xlab='y', ylab='F_y', main = 'Posterior predictive ECDF')
temp = sapply(1:nrow(fit$post_ypred), function(s)
  lines(y0, ecdf(fit$post_ypred[s,])(y0), # ECDF of posterior predictive draws
        col='gray', type ='s'))
lines(y0, ecdf(y_test)(y0),  # ECDF of testing data
     col='black', type = 's', lwd = 3)

# The posterior predictive checks usually do not pass!
# try ?sbqr instead...

}
