#' Sample size calculator for estimation of population mean under SRS
#'
#' @description
#' The SscSrsMean function calculates the sample size required for estimation of population mean
#' based upon the availability of prior information on coefficient of variation (CV) of the
#' population under Simple Random Sampling (SRS) with or without replacement sampling design for
#' a given confidence level and relative error. Further, the function provides the required sample size
#' when population CV is known whereas it provides both required sample size and additional sample units
#' to be surveyed when population CV is unknown under both SRS with or without replacement.
#'
#' @param CV_known Logical. TRUE if CV of the population is known; FALSE otherwise.
#' @param replace Logical. TRUE for SRSWR; FALSE for SRSWOR.
#' @param alpha Level of significance (e.g., 0.05 for 5\% level).
#' @param re Relative error in the estimate (e.g., 0.1 for 10\%).
#' @param N Population size. Use NA for SRSWR.
#' @param preliminary_sample Numeric vector. Preliminary sample data; NA if CV is known.
#' @param cv_value Numeric. Known CV when CV_known = TRUE; ignored if CV_known = FALSE.
#'
#' @return A list with the following components:
#' \describe{
#'   \item{Required sample size}{The required sample size for the given inputs.}
#'   \item{Additional sample units to be surveyed}{Only for unknown CV: additional units beyond preliminary sample size.}
#' }
#'
#' @details
#' Returns the sample size for estimation of the population mean under SRS with or without replacement.
#' For unknown CV, also returns the number of extra units needed beyond the preliminary sample.
#'
#' @references
#' Cochran, W. G. (1977). \emph{Sampling Techniques, 3rd Edition}. New York: John Wiley & Sons.
#'
#' Singh, D. and Chaudhary, F.S. (1986). \emph{Theory and Analysis of Sample Survey Designs}. New York: John Wiley & Sons, Inc.
#'
#' Sukhatme, P.V., Sukhatme, B.V., Sukhatme, S. and Asok, C. (1984). \emph{Sampling Theory of Surveys with Applications}. Iowa State University Press, Ames and Indian Society of Agricultural Statistics, New Delhi.
#'
#' @examples
#' # Known CV, SRSWOR
#' SscSrsMean(TRUE, FALSE, 0.05, 0.2, 100, NA, cv_value = 0.3)
#'
#' # Unknown CV, SRSWOR
#' prelim <- c(12, 14, 8, 5, 36, 24, 18, 17, 6, 9)
#' SscSrsMean(FALSE, FALSE, 0.05, 0.2, 100, prelim)
#'
#' # Known CV, SRSWR
#' SscSrsMean(TRUE, TRUE, 0.05, 0.2, NA, NA, cv_value = 0.3)
#'
#' @export
#' @importFrom stats qnorm qt sd
SscSrsMean <- function(CV_known, replace, alpha, re, N, preliminary_sample, cv_value = NA) {

  # CV known, SRSWOR
  if (CV_known == TRUE && replace == FALSE) {
    if (is.na(cv_value)) stop("cv_value must be provided when CV_known = TRUE")
    cv <- cv_value
    Z <- qnorm(alpha / 2, lower.tail = FALSE)
    size <- round((N * cv^2) / (cv^2 + (N * re^2 / Z^2)), 0)
    output <- list("Required sample size" = size)
    return(output)
  }

  # CV unknown, SRSWOR
  if (CV_known == FALSE && replace == FALSE) {
    cv <- sd(preliminary_sample) / mean(preliminary_sample)
    t <- qt(alpha / 2, (length(preliminary_sample) - 1), lower.tail = FALSE)
    size <- round((N * cv^2) / (cv^2 + (N * re^2 / t^2)), 0)
    additional_size <- size - length(preliminary_sample)
    output <- list(
      "Required sample size" = size,
      "Additional sample units to be surveyed" = additional_size
    )
    return(output)
  }

  # CV known, SRSWR
  if (CV_known == TRUE && replace == TRUE) {
    if (is.na(cv_value)) stop("cv_value must be provided when CV_known = TRUE")
    cv <- cv_value
    Z <- qnorm(alpha / 2, lower.tail = FALSE)
    size <- round(((Z * cv) / re)^2, 0)
    output <- list("Required sample size" = size)
    return(output)
  }

  # CV unknown, SRSWR
  if (CV_known == FALSE && replace == TRUE) {
    cv <- sd(preliminary_sample) / mean(preliminary_sample)
    t <- qt(alpha / 2, (length(preliminary_sample) - 1), lower.tail = FALSE)
    size <- round(((t * cv) / re)^2, 0)
    additional_size <- size - length(preliminary_sample)
    output <- list(
      "Required sample size" = size,
      "Additional sample units to be surveyed" = additional_size
    )
    return(output)
  }
}
