#' Generate Geometric Temperature Ladder for Parallel Tempering
#'
#' Produces a geometric progression of inverse-temperatures (or temperatures)
#' commonly used in parallel-tempered MCMC algorithms.
#'
#' @param N Integer > 1, number of chains/temperatures.
#' @param T1 Numeric > 0, coldest temperature (usually 1).
#' @param TN Numeric > T1, hottest temperature.
#'
#' @return Numeric vector of length N containing the geometrically spaced
#'         temperatures T_1, T_2, ..., T_N.
#'
#' @examples
#' geoTemp(8, T1 = 1, TN = 20)
#'
#' @export
geoTemp <- function(N, T1 = 1, TN = 20) {
  if (N <= 1) stop("N must be > 1")
  if (T1 <= 0) stop("T1 must be > 0")
  if (TN <= T1) stop("TN must be > T1")
  
  T1 * (TN / T1)^((0:(N - 1)) / (N - 1))
}
