% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_K_trait.R
\name{simulate_K_trait}
\alias{simulate_K_trait}
\title{Simulate Trait Data with Target Phylogenetic Signal (Blomberg's K)}
\usage{
simulate_K_trait(target_K, tree, max_attempts = 1e+05, tolerance = 0.02)
}
\arguments{
\item{target_K}{Numeric. The desired phylogenetic signal strength (K value).
- K = 0: No phylogenetic signal (star phylogeny)
- K = 1: Expected signal under Brownian motion evolution
- K > 1: Stronger phylogenetic signal than expected under Brownian motion
- 0 < K < 1: Weaker phylogenetic signal than expected under Brownian motion}

\item{tree}{An object of class "phylo". The phylogenetic tree for trait simulation.}

\item{max_attempts}{Integer. Maximum number of simulation attempts before giving up.
Default is 100000.}

\item{tolerance}{Numeric. Acceptable difference between target and estimated K.
Default is 0.02.}
}
\value{
A data.frame with one column named 'trait' containing the simulated trait values.
  Row names correspond to tip labels from the phylogenetic tree. Returns NULL if
  the target K cannot be achieved within the specified tolerance and attempts.
}
\description{
This function generates trait data that matches a specified phylogenetic signal
strength (Blomberg's K) through iterative simulation and testing.
}
\details{
The function works by:

1. Transforming the phylogenetic tree according to the target K value

2. Simulating trait data using phytools::fastBM() on the transformed tree

3. Estimating the phylogenetic signal using phytools::phylosig()

4. Repeating until the estimated K is within tolerance of the target

Tree transformation strategies:
- When target_K = 0: Creates a star phylogeny using ape::stree()
- When target_K = 1: Uses the original tree without transformation
- When target_K > 1: Scales all branch lengths by the target K value
- When 0 < target_K < 1: Interpolates between original tree and uniform branch lengths
}
\note{
Blomberg's K measures the strength of phylogenetic signal relative to what would
be expected under a Brownian motion model of evolution. Unlike Pagel's lambda,
K can exceed 1, indicating stronger phylogenetic clustering than expected.

The function may take considerable time to converge for certain K values.
Consider adjusting the tolerance parameter if convergence is slow.
}
\examples{
# Generate a random tree
tree <- ape::rtree(50)

# Simulate trait with expected Brownian motion signal
trait_data <- simulate_K_trait(0.9, tree)

# Verify the phylogenetic signal
trait_vector <- setNames(trait_data$trait, rownames(trait_data))
phytools::phylosig(tree, trait_vector, method = "K", test = TRUE)

}
