# ExpoInvChen.R
#' Exponentiated Inverse Chen Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Exponentiated Inverse Chen distribution.
#'
#' The Exponentiated Inverse Chen distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Exponentiated Inverse Chen distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta)
#' = 1 - \left[ 1 - \exp\left( \lambda \left( 1 - \exp(x^{-\alpha}) \right) \right) \right]^{\theta},
#' \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dexpo.inv.chen()} — Density function
#'   \item \code{pexpo.inv.chen()} — Distribution function
#'   \item \code{qexpo.inv.chen()} — Quantile function
#'   \item \code{rexpo.inv.chen()} — Random generation
#'   \item \code{hexpo.inv.chen()} — Hazard function
#' }
#'
#' @usage
#' dexpo.inv.chen(x, alpha, lambda, theta, log = FALSE)
#' pexpo.inv.chen(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qexpo.inv.chen(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rexpo.inv.chen(n, alpha, lambda, theta)
#' hexpo.inv.chen(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dexpo.inv.chen}: numeric vector of (log-)densities
#'   \item \code{pexpo.inv.chen}: numeric vector of probabilities
#'   \item \code{qexpo.inv.chen}: numeric vector of quantiles
#'   \item \code{rexpo.inv.chen}: numeric vector of random variates
#'   \item \code{hexpo.inv.chen}: numeric vector of hazard values
#' }
#'
#' @references   
#' Telee, L. B. S., & Kumar, V. (2023). 
#' Exponentiated Inverse Chen distribution: Properties and applications. 
#' \emph{Journal of Nepalese Management Academia}, \bold{1(1)}, 53--62. 
#' \doi{10.3126/jnma.v1i1.62033}
#'
#' Srivastava, A.K., & Kumar, V.(2011). 
#' Markov Chain Monte Carlo Methods for Bayesian Inference of the Chen Model. 
#' \emph{International Journal of Computer Information Systems}, \bold{2(2)}, 7--14. 
#'
#' @examples
#' x <- seq(2, 5, 0.25)
#' dexpo.inv.chen(x, 0.5, 2.5, 1.5)
#' pexpo.inv.chen(x, 0.5, 2.5, 1.5)
#' qexpo.inv.chen(0.5, 0.5, 2.5, 1.5)
#' rexpo.inv.chen(10, 0.5, 2.5, 1.5)
#' hexpo.inv.chen(x, 0.5, 2.5, 1.5)
#' 
#' # Data
#' x <- headneck44
#' # ML estimates
#' params = list(alpha=0.3947, lambda=15.5330, theta=8.1726)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pexpo.inv.chen, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qexpo.inv.chen, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' # Display plot and print numerical summary
#' gofic(x, params = params,
#'       dfun = dexpo.inv.chen, pfun=pexpo.inv.chen, plot=TRUE, verbose = TRUE)
#' 
#' @name ExpoInvChen
#' @aliases dexpo.inv.chen pexpo.inv.chen qexpo.inv.chen rexpo.inv.chen hexpo.inv.chen
#' @export
NULL

#' @rdname ExpoInvChen
#' @usage NULL
#' @export   
dexpo.inv.chen <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
    u <- exp(x[valid] ^ - alpha)
	z <- exp(lambda*(1-u))
    pdf[valid] <- alpha*lambda*theta*(x[valid]^-(alpha +1.0))*u*z*((1-z)^(theta-1))
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname ExpoInvChen
#' @usage NULL
#' @export
pexpo.inv.chen <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- exp(q[valid] ^ (-alpha))
	z <- exp(lambda * (1-u))
    cdf[valid] <- 1.0 - (1.0 - z) ^ theta
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
 }

#' @rdname ExpoInvChen
#' @usage NULL
#' @export
qexpo.inv.chen <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
  	qtl[valid] <-  (log(1-(1/lambda)*log(1-(1-p[valid])^(1/theta))))^ (-1/alpha) 
	}

    return(qtl)
}

#' @rdname ExpoInvChen
#' @usage NULL
#' @export
rexpo.inv.chen <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qexpo.inv.chen(u, alpha, lambda, theta)
    return(q)
}

#' @rdname ExpoInvChen
#' @usage NULL
#' @export
hexpo.inv.chen <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0))
		stop("Invalid arguments: alpha, lambda, theta, and x must be positive.")  
	 
	nume <- dexpo.inv.chen(x, alpha, lambda, theta) 
	surv <- 1.0 - pexpo.inv.chen(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 



