#' Carry out Little's test of MCAR.
#'
#' @param X The dataset with incomplete data, where all the pairs of variables are observed together.
#' @param type Determines the test statistic to use, based on the discussion in Section 6 in \insertCite{BB2024;textual}{MCARtest}.
#' The default option is "mean&cov", and uses the test statistic \eqn{d^2_{\mathrm{aug}}}. When set equal to "cov", implements a test
#' of MCAR based on \eqn{d^2_{\mathrm{cov}}}, while, when set equal to "mean", implements the classical Little's test as defined in
#' \insertCite{Little1988;textual}{MCARtest}.
#'
#' @references \insertRef{BB2024}{MCARtest}
#' @references \insertRef{Little1988}{MCARtest}
#'
#' @return  The p-value of Little's test, found by comparing the log likelihood ratio statistic
#' to the chi-squared distribution with the appropriate number
#' of degrees of freedom. Described in \insertCite{Little1988;textual}{MCARtest}.
#' @export
#'
#' @importFrom norm prelim.norm em.norm getparam.norm
#' @importFrom stats cor qchisq
#'
#' @examples
#' library(MASS)
#' n = 200
#'
#' SigmaS=list() #Random 2x2 correlation matrices (necessarily consistent)
#' for(j in 1:3){
#' x=runif(2,min=-1,max=1); y=runif(2,min=-1,max=1)
#' SigmaS[[j]]=cov2cor(x%*%t(x) + y%*%t(y))
#' }
#'
#' X1 = mvrnorm(n, c(0,0), SigmaS[[1]])
#' X2 = mvrnorm(n, c(0,0), SigmaS[[2]])
#' X3 = mvrnorm(n, c(0,0), SigmaS[[3]])
#' columns = c("X1","X2","X3")
#' X = data.frame(matrix(nrow = 3*n, ncol = 3))
#' X[1:n, c("X1", "X2")] = X1
#' X[(n+1):(2*n), c("X2", "X3")] = X2
#' X[(2*n+1):(3*n), c("X1", "X3")] = X3
#' X = as.matrix(X)
#'
#' little_test(X)
#' little_test(X, type = "mean&cov")
#' little_test(X, type = "mean")


little_test = function(X, type="mean&cov"){
  s = prelim.norm(as.matrix(X))
  thetahat = em.norm(s)
  mu_true = getparam.norm(s,thetahat,corr=FALSE)$mu
  Sigma_true = getparam.norm(s,thetahat,corr=FALSE)$sigma
  
  result = get_SigmaS(X, min_diff = 0)
  Omega_S = result$C_S
  patterns = result$pattern; data_pattern = result$data_pattern
  n_pattern = result$n_pattern; d = result$ambient_dimension
  
  if(type == "mean"){
    d_squared = 0
    df = -d
    
    for (i in 1:n_pattern){
      n_S = dim(data_pattern[[i]])[1]
      card_S = dim(data_pattern[[i]])[2]
      
      x_S = colMeans(data_pattern[[i]]) - mu_true[patterns[[i]]]
      L_S = as.matrix(Sigma_true[patterns[[i]], patterns[[i]]])
      
      d_squared = d_squared + n_S*t(x_S)%*%solve(L_S)%*%x_S
      df = df + card_S
    }
    p_L = pchisq(d_squared, df, lower.tail = FALSE)
  }
  
  else if(type == "cov"){
    
    d_cov = 0
    df = -d*(d+1)/2
    
    for (i in 1:n_pattern){
      n_S = dim(data_pattern[[i]])[1]
      card_S = dim(data_pattern[[i]])[2]
      
      x_S = colMeans(data_pattern[[i]]) - mu_true[patterns[[i]]]
      L_S = as.matrix(Sigma_true[patterns[[i]], patterns[[i]]])
      
      d_cov = d_cov + n_S*(sum(diag(Omega_S[[i]]%*%solve(L_S))) - card_S -
                             log(det(Omega_S[[i]])) + log(det(L_S)))
      df = df + card_S*(card_S+1)/2
      print(df)
    }
    
    p_L = pchisq(d_cov, df, lower.tail = FALSE)
  }
  
  else{
    
    d_aug = 0
    df = -d*(d+3)/2
    
    for (i in 1:n_pattern){
      n_S = dim(data_pattern[[i]])[1]
      card_S = dim(data_pattern[[i]])[2]
      
      x_S = colMeans(data_pattern[[i]]) - mu_true[patterns[[i]]]
      L_S = as.matrix(Sigma_true[patterns[[i]], patterns[[i]]])
      
      d_aug = d_aug + n_S*t(x_S)%*%solve(L_S)%*%x_S + 
        n_S*(sum(diag(Omega_S[[i]]%*%solve(L_S))) - card_S -
               log(det(Omega_S[[i]])) + log(det(L_S)))
      df = df + card_S*(card_S+3)/2
    }
    
    p_L = pchisq(d_aug, df, lower.tail = FALSE)
  }
  
  return(p_L)
  
}


