#' @title S3 Methods: update
#'
#' @description
#' The \code{update} function provides a unified and convenient interface to refresh or modify
#' existing objects generated by the \code{LCPA} package. It allows users to re-run model fitting
#' or data simulation with new parameter settings while preserving all other original configurations.
#' Supported classes include: \code{\link[LCPA]{LCA}}, \code{\link[LCPA]{LPA}},
#' \code{\link[LCPA]{LCPA}}, \code{\link[LCPA]{LTA}}, \code{\link[LCPA]{sim.LCA}},
#' \code{\link[LCPA]{sim.LPA}}, and \code{\link[LCPA]{sim.LTA}}.
#'
#' @param x An object of one of the following classes:
#'   \itemize{
#'     \item \code{\link[LCPA]{LCA}} — Latent Class Analysis model.
#'     \item \code{\link[LCPA]{LPA}} — Latent Profile Analysis model.
#'     \item \code{\link[LCPA]{LCPA}} — Latent Class Prediction Analysis (with covariates).
#'     \item \code{\link[LCPA]{LTA}} — Latent Transition Analysis model.
#'     \item \code{\link[LCPA]{sim.LCA}} — Simulated LCA dataset.
#'     \item \code{\link[LCPA]{sim.LPA}} — Simulated LPA dataset.
#'     \item \code{\link[LCPA]{sim.LTA}} — Simulated LTA dataset.
#'   }
#' @param ... Additional named arguments passed to override or extend the original call.
#'   Valid arguments depend on the class of \code{x}:
#'   \describe{
#'     \item{\strong{\code{LCA}}}{\code{response}, \code{L}, \code{par.ini}, \code{method},
#'       \code{is.sort}, \code{nrep}, \code{vis}, \code{control.EM}, \code{control.Mplus}, \code{control.NNE}}
#'     \item{\strong{\code{LPA}}}{\code{response}, \code{L}, \code{par.ini}, \code{constraint},
#'       \code{method}, \code{is.sort}, \code{nrep}, \code{vis}, \code{control.EM}, \code{control.Mplus}, \code{control.NNE}}
#'     \item{\strong{\code{LCPA}}}{\code{response}, \code{L}, \code{ref.class}, \code{type}, \code{covariates},
#'       \code{CEP.error}, \code{par.ini}, \code{params}, \code{is.sort}, \code{constraint}, \code{method},
#'       \code{tol}, \code{maxiter}, \code{nrep}, \code{starts}, \code{maxiter.wa}, \code{vis},
#'       \code{control.EM}, \code{control.Mplus}, \code{control.NNE}}
#'     \item{\strong{\code{LTA}}}{\code{responses}, \code{L}, \code{ref.class}, \code{type}, \code{covariates},
#'       \code{CEP.timeCross}, \code{CEP.error}, \code{covariates.timeCross}, \code{par.ini}, \code{params},
#'       \code{is.sort}, \code{constraint}, \code{method}, \code{tol}, \code{maxiter}, \code{nrep}, \code{starts},
#'       \code{maxiter.wa}, \code{vis}, \code{control.EM}, \code{control.Mplus}, \code{control.NNE}}
#'     \item{\strong{\code{sim.LCA}}}{\code{N}, \code{I}, \code{L}, \code{poly.value}, \code{IQ},
#'       \code{distribution}, \code{params}, \code{is.sort}}
#'     \item{\strong{\code{sim.LPA}}}{\code{N}, \code{I}, \code{L}, \code{constraint}, \code{distribution},
#'       \code{mean.range}, \code{covs.range}, \code{params}, \code{is.sort}}
#'     \item{\strong{\code{sim.LTA}}}{\code{N}, \code{I}, \code{L}, \code{times}, \code{type}, \code{rate},
#'       \code{constraint}, \code{distribution}, \code{mean.range}, \code{covs.range}, \code{poly.value},
#'       \code{IQ}, \code{covariates}, \code{beta}, \code{gamma}, \code{params}, \code{is.sort}}
#'   }
#'
#' @return An object of the same class as \code{x}, reconstructed using the original arguments
#'   updated with any provided in \code{...}. All unchanged parameters are preserved from the original call.
#'
#' @details
#' Internally, each method extracts the stored \code{arguments} list from the input object \code{x},
#' merges it with user-provided \code{...} using \code{\link[utils]{modifyList}}, then re-invokes
#' the corresponding constructor function (\code{LCA()}, \code{LPA()}, \code{LCPA()}, \code{LTA()},
#' \code{sim.LCA()}, etc.) with the merged argument list.
#'
#' This ensures that:
#' \itemize{
#'   \item Only explicitly overridden parameters are changed.
#'   \item Default values from the original call remain intact.
#'   \item Complex nested structures (e.g., control lists) can be partially updated.
#' }
#'
#' Note: If an invalid argument is passed (e.g., \code{constraint} to \code{LCA}), it will be silently ignored
#' unless the underlying constructor validates inputs.
#'
#' @examples
#' \donttest{
#' library(LCPA)
#'
#' # --- Update LCA ---
#' data <- sim.LCA(N=500, I=5, L=3)
#' lca.obj <- LCA(data$response, L=3)
#' lca.updated <- update(lca.obj, method="EM", nrep=5)
#'
#' # --- Update LPA ---
#' data2 <- sim.LPA(N=300, I=4, L=2)
#' lpa.obj <- LPA(data2$response, L=2, constraint="VE")
#' lpa.updated <- update(lpa.obj, constraint="VV")
#'
#' # --- Update Simulation Objects ---
#' sim.obj1 <- sim.LCA(N=1000)
#' sim.obj1_updated <- update(sim.obj1, N=2000, IQ=0.8)
#'
#' sim.obj2 <- sim.LPA(I=6)
#' sim.obj2_updated <- update(sim.obj2, I=8, mean.range=c(-2,2))
#'
#' sim.obj3 <- sim.LTA(N=200, I=5, L=2, times=3)
#' sim.obj3_updated <- update(sim.obj3, N=300, times=4, constraint="ER")
#' }
#'
#' @name update
update <- function(x, ...) {
  UseMethod("update")
}

#' @describeIn update Update method for \code{LCA} objects
#' @importFrom utils modifyList
#' @export
update.LCA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))

  LCA.updated <- LCA(
    response = arguments.new$response,
    L = arguments.new$L,
    par.ini = arguments.new$par.ini,
    method = arguments.new$method,
    is.sort = arguments.new$is.sort,
    nrep = arguments.new$nrep,
    vis = arguments.new$vis,
    control.EM = arguments.new$control.EM,
    control.Mplus = arguments.new$control.Mplus,
    control.NNE = arguments.new$control.NNE
  )
  return(LCA.updated)
}

#' @describeIn update Update method for \code{LPA} objects
#' @importFrom utils modifyList
#' @export
update.LPA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))

  LPA.updated <- LPA(
    response = arguments.new$response,
    L = arguments.new$L,
    par.ini = arguments.new$par.ini,
    constraint = arguments.new$constraint,
    method = arguments.new$method,
    is.sort = arguments.new$is.sort,
    nrep = arguments.new$nrep,
    vis = arguments.new$vis,
    control.EM = arguments.new$control.EM,
    control.Mplus = arguments.new$control.Mplus,
    control.NNE = arguments.new$control.NNE
  )
  return(LPA.updated)
}

#' @describeIn update Update method for \code{LCPA} objects
#' @importFrom utils modifyList
#' @export
update.LCPA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))

  LCPA.updated <- LCPA(
    response = arguments.new$response,
    L = arguments.new$L,
    ref.class = arguments.new$ref.class,
    type = arguments.new$type,
    covariate = arguments.new$covariate,
    CEP.error = arguments.new$CEP.error,
    par.ini = arguments.new$par.ini,
    params = arguments.new$params,
    is.sort = arguments.new$is.sort,
    constraint = arguments.new$constraint,
    method = arguments.new$method,
    tol = arguments.new$tol,
    method.SE=arguments.new$method.SE,
    n.Bootstrap=arguments.new$n.Bootstrap,
    maxiter = arguments.new$maxiter,
    nrep = arguments.new$nrep,
    starts = arguments.new$starts,
    maxiter.wa = arguments.new$maxiter.wa,
    vis = arguments.new$vis,
    control.EM = arguments.new$control.EM,
    control.Mplus = arguments.new$control.Mplus,
    control.NNE = arguments.new$control.NNE
  )

  return(LCPA.updated)
}

#' @describeIn update Update method for \code{LTA} objects
#' @importFrom utils modifyList
#' @export
update.LTA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))

  LTA.updated <- LTA(
    responses = arguments.new$responses,
    L = arguments.new$L,
    ref.class = arguments.new$ref.class,
    type = arguments.new$type,
    covariates = arguments.new$covariates,
    CEP.timeCross = arguments.new$CEP.timeCross,
    CEP.error = arguments.new$CEP.error,
    covariates.timeCross = arguments.new$covariates.timeCross,
    par.ini = arguments.new$par.ini,
    params = arguments.new$params,
    is.sort = arguments.new$is.sort,
    constraint = arguments.new$constraint,
    method = arguments.new$method,
    tol = arguments.new$tol,
    method.SE=arguments.new$method.SE,
    n.Bootstrap=arguments.new$n.Bootstrap,
    maxiter = arguments.new$maxiter,
    nrep = arguments.new$nrep,
    starts = arguments.new$starts,
    maxiter.wa = arguments.new$maxiter.wa,
    vis = arguments.new$vis,
    control.EM = arguments.new$control.EM,
    control.Mplus = arguments.new$control.Mplus,
    control.NNE = arguments.new$control.NNE
  )

  return(LTA.updated)
}

#' @describeIn update Update method for \code{sim.LCA} objects
#' @importFrom utils modifyList
#' @export
update.sim.LCA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))

  sim.updated <- sim.LCA(
    N = arguments.new$N,
    I = arguments.new$I,
    L = arguments.new$L,
    poly.value = arguments.new$poly.value,
    IQ = arguments.new$IQ,
    distribution = arguments.new$distribution,
    params = arguments.new$params,
    is.sort=arguments.new$is.sort
  )
  return(sim.updated)
}

#' @describeIn update Update method for \code{sim.LPA} objects
#' @importFrom utils modifyList
#' @export
update.sim.LPA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))

  sim.updated <- sim.LPA(
    N = arguments.new$N,
    I = arguments.new$I,
    L = arguments.new$L,
    constraint = arguments.new$constraint,
    distribution = arguments.new$distribution,
    mean.range = arguments.new$mean.range,
    covs.range = arguments.new$covs.range,
    params = arguments.new$params,
    is.sort=arguments.new$is.sort
  )
  return(sim.updated)
}

#' @describeIn update Update method for \code{sim.LTA} objects
#' @importFrom utils modifyList
#' @export
update.sim.LTA <- function(x, ...) {
  arguments.current <- x$arguments

  arguments.new <- modifyList(arguments.current, list(...))



  sim.updated <- sim.LTA(
    N = arguments.new$N,
    I = arguments.new$I,
    L = arguments.new$L,
    distribution = arguments.new$distribution,
    times = arguments.new$times,
    type = arguments.new$type,
    rate = arguments.new$rate,
    constraint = arguments.new$constraint,
    mean.range = arguments.new$mean.range,
    covs.range = arguments.new$covs.range,
    poly.value = arguments.new$poly.value,
    IQ = arguments.new$IQ,
    params = arguments.new$params,
    is.sort=arguments.new$is.sort,
    covariates = arguments.new$covariates,
    beta = arguments.new$beta,
    gamma = arguments.new$gamma
  )
  return(sim.updated)
}
