% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fMRI.R
\name{fmri_changepoints}
\alias{fmri_changepoints}
\title{Detect changepoints in a resting state fMRI scan}
\usage{
fmri_changepoints(data, p = 100, k = 0.3)
}
\arguments{
\item{data}{A four dimensional array, where the fourth dimension is time.}

\item{p}{Number of random vector projections, set to 100 by default.}

\item{k}{Penalty constant passed to pruned exact linear time algorithm.}
}
\value{
A list consisting of:
\itemize{
\item \verb{$changepoints} : Indices of the change-points detected; will return \code{integer(0)} if no changepoints are detected.
\item \verb{$method} : A \code{string} \code{"fMRI changepoints (KWCChangepoint)"}
}
}
\description{
Functional magnetic resonance imaging scans are expected to be
stationary after being pre-processed. This function attempts to find
potential changepoints using the findings of Ramsay and Chenouri (2025).
}
\note{
The penalty is of the form \deqn{3.74 + k\sqrt{n}} where \eqn{n} is the
number of observations. In the case that there is potentially correlated
observations, the parameter could be set to \eqn{k=1}. More information could
be found in the reference.

The example in this document is a simple "toy example", as good fMRI data
simulation requires more dependencies. For generating fMRI data, see
\code{\link[neuRosim:simVOLfmri]{neuRosim::simVOLfmri()}}, \code{\link[neuRosim:simTSrestingstate]{neuRosim::simTSrestingstate()}}.
}
\examples{
# In order to replicate how a changepoint would appear in a resting-state
# fMRI scan in a manner that is not computationally expensive, this example
# constructs an image of a 3D ball taken at 12 time stamps. The noise, and
# therefore the covariance function, changes at time stamp 6.
x_dim <- 24
y_dim <- 24
z_dim <- 10
time_dim <- 12
image_array <- array(0, dim = c(x_dim, y_dim, z_dim, time_dim))

center <- c(x_dim / 2, y_dim / 2, z_dim / 2)
radius <- min(x_dim, y_dim, z_dim) / 4

set.seed(42)

for (t in 1:time_dim) {
  for (x in 1:x_dim) {
    for (y in 1:y_dim) {
      for (z in 1:z_dim) {
        dist_from_center <- sqrt((x - center[1])^2 + (y - center[2])^2 + (z - center[3])^2)
        if (dist_from_center <= radius) {
          # Adding noise with increasing variability at timestamp 6
          if (t <= 6) {
            noise <- rnorm(1, mean = 0, sd = 0.1)  # Low variability noise
          } else {
            noise <- rnorm(1, mean = 0, sd = 2)  # High variability noise
          }
          image_array[x, y, z, t] <- noise
        } else {
          # Add lower intensity noise outside the ball
          image_array[x, y, z, t] <- rnorm(1, mean = 0, sd = 0.005)
        }
      }
    }
  }
}
fmri_changepoints(image_array, k = 0.1, p = 10)

}
\references{
Ramsay, K., & Chenouri, S. (2025). Robust changepoint detection
in the variability of multivariate functional data. Journal of
Nonparametric Statistics. https://doi.org/10.1080/10485252.2025.2503891
}
