# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_cpi

library(testthat)

# ============================================================
# 1. Estructura general del tibble
# ============================================================
test_that("get_japan_cpi() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_cpi()

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

# ============================================================
# 2. Dimensiones esperadas y años correctos
# ============================================================
test_that("get_japan_cpi() returns correct dimensions and years", {
  skip_on_cran()

  result <- get_japan_cpi()

  # Expected number of rows (2010 to 2022 inclusive)
  expect_equal(nrow(result), 13)

  # Years must match exactly 2010:2022
  expect_equal(sort(result$year), 2010:2022)

  # Data should be sorted in descending order
  expect_true(all(diff(result$year) <= 0))
})

# ============================================================
# 3. Consistencia interna de valores
# ============================================================
test_that("get_japan_cpi() returns consistent values for Japan", {
  skip_on_cran()

  result <- get_japan_cpi()

  # Country must always be Japan
  expect_true(all(result$country == "Japan"))

  # Indicator must always be CPI
  expect_true(all(grepl("Consumer price index", result$indicator, ignore.case = TRUE)))

  # All CPI values must be positive
  expect_true(all(result$value > 0, na.rm = TRUE))

  # CPI values must be reasonable (0–300)
  expect_true(all(result$value > 0 & result$value < 300, na.rm = TRUE))
})

# ============================================================
# 4. Manejo correcto de errores de API
# ============================================================
test_that("get_japan_cpi() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_cpi(), error = function(e) NULL)

  # Must return NULL or tibble but never crash
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
