% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Inspect.R
\name{Inspect}
\alias{Inspect}
\title{Informative sparse projection for estimating change-points (Inspect)}
\usage{
Inspect(
  X,
  lambda = NULL,
  xi = NULL,
  alpha = 1.5,
  K = 5,
  eps = 1e-10,
  empirical = FALSE,
  maxiter = 10000,
  N = 100,
  tol = 1/100,
  rescale_variance = TRUE,
  debug = FALSE
)
}
\arguments{
\item{X}{Matrix of observations, where each row contains a time series}

\item{lambda}{Manually specified value of \eqn{\lambda} (can be \code{NULL}, in which case \eqn{\lambda \gets \sqrt{\log(p\log n)/2}})}

\item{xi}{Manually specified value of \eqn{\xi} (can be NULL, in which case \eqn{\xi \gets 4\sqrt{\log(np)}})}

\item{alpha}{Parameter for generating seeded intervals}

\item{K}{Parameter for generating seeded intervals}

\item{eps}{Threshold for declaring numerical convergence of the power method}

\item{empirical}{If \code{TRUE}, the detection threshold \eqn{xi} is based on Monte Carlo simulation using \code{\link{Inspect_calibrate}}}

\item{maxiter}{Maximum number of iterations for the power method}

\item{N}{If \code{empirical=TRUE}, \code{N} is the number of Monte Carlo samples used}

\item{tol}{If \code{empirical=TRUE}, \code{tol} is the false error probability tolerance}

\item{rescale_variance}{If \code{TRUE}, each row of the data is re-scaled by a MAD estimate using \code{\link{rescale_variance}}}

\item{debug}{If \code{TRUE}, diagnostic prints are provided during execution}
}
\value{
A list containing 
  \item{changepoints}{vector of estimated change-points}
  \item{changepointnumber}{number of changepoints}
  \item{CUSUMval}{vector with the sparse projected CUSUMs corresponding to \code{changepoints}}
  \item{coordinates}{a matrix of zeros and ones indicating which time series are affected by a change in mean, with each row corresponding to the change-point in \code{changepoints}}
  \item{scales}{vector of estimated noise level for each series}
}
\description{
R wrapper for C function implementing a Narrowest-Over-Threshold variant of Inspect \insertCite{wang_high_2018;textual}{HDCD} as specified in Appendix C in \insertCite{moen2023efficient;textual}{HDCD}. Note that the algorithm is only implemented for \eqn{\mathcal{S} = \mathcal{S}_2}, in the notation of \insertCite{moen2023efficient;textual}{HDCD}.
}
\examples{
library(HDCD)
n = 50
p = 50
set.seed(100)
# Generating data
X = matrix(rnorm(n*p), ncol = n, nrow=p)
# Adding a single sparse change-point:
X[1:5, 26:n] = X[1:5, 26:n] +1

# Vanilla Inspect:
res = Inspect(X)
res$changepoints

# Manually setting leading constants for \lambda(t) and \gamma(t)
res = Inspect(X, 
              lambda = sqrt(log(p*log(n))/2),
              xi = 4*sqrt(log(n*p))
)
res$changepoints #estimated change-point locations

# Empirical choice of thresholds:
res = Inspect(X, empirical=TRUE, N = 100, tol = 1/100)
res$changepoints

# Manual empirical choice of thresholds (equivalent to the above)
thresholds_emp = Inspect_calibrate(n,p, N=100, tol=1/100)
res = Inspect(X, xi = thresholds_emp$max_value)
res$changepoints
}
\references{
\insertAllCited{}
}
