% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gem_4_4.R
\name{gem_4_4}
\alias{gem_4_4}
\title{Some Simple 4-by-4 General Equilibrium Models}
\usage{
gem_4_4(...)
}
\arguments{
\item{...}{arguments to be passed to the function sdm2.}
}
\description{
Some simple 4-by-4 general equilibrium models.
}
\examples{
\donttest{
#### A general equilibrium model containing a capital good with service-life
# wear-and-tear. The new product can be used as a capital good with a service
# life of two periods, and the used old capital good is the old product.
ge <- sdm2(
  A = function(state) {
    a.firm1 <- CD_A(alpha = 2, Beta = c(0, 0.5, 0.5, 0), state$p)
    a.consumer <- c(1, 0, 0, 0)
    a.firm2 <- c(1, 0, 0, 0)
    a.firm3 <- c(0, 0, 0, 1)
    cbind(a.firm1, a.consumer, a.firm2, a.firm3)
  },
  B = matrix(c(
    1, 0, 0, 0,
    0, 0, 1, 1,
    0, 0, 0, 0,
    0, 0, 1, 0
  ), 4, 4, TRUE),
  S0Exg = matrix(c(
    NA, NA, NA, NA,
    NA, NA, NA, NA,
    NA, 100, NA, NA,
    NA, NA, NA, NA
  ), 4, 4, TRUE),
  names.commodity = c("prod.new", "cap", "lab", "prod.old"),
  names.agent = c("firm1", "consumer", "firm2", "firm3"),
  numeraire = "prod.new",
  priceAdjustmentVelocity = 0.05
)

ge$p
ge$z
addmargins(ge$D, 2)
addmargins(ge$S, 2)

#### the Shoven-Whalley model at
## https://lexjansen.com/nesug/nesug03/st/st002.pdf
ge <- sdm2(
  A = function(state) {
    a.firm.corn <- CES_A(sigma = 1 - 1 / 2, alpha = 1.5, Beta = c(0, 0, 0.4, 0.6), state$p)
    a.firm.iron <- CES_A(sigma = 1 - 1 / 0.5, alpha = 2, Beta = c(0, 0, 0.3, 0.7), state$p)
    a.consumer1 <- SCES_A(alpha = 1, Beta = c(0.5, 0.5, 0, 0), es = 1.5, p = state$p)
    a.consumer2 <- SCES_A(alpha = 1, Beta = c(0.3, 0.7, 0, 0), es = 0.75, p = state$p)

    cbind(a.firm.corn, a.firm.iron, a.consumer1, a.consumer2)
  },
  B = diag(c(1, 1, 0, 0), 4, 4),
  S0Exg = {
    tmp <- matrix(NA, 4, 4)
    tmp[3, 3] <- 25
    tmp[4, 4] <- 60
    tmp
  },
  names.commodity = c("corn", "iron", "cap", "lab"),
  names.agent = c("firm.corn", "firm.iron", "consumer1", "consumer2"),
  numeraire = "lab"
)

ge$p
ge$z
ge$D
ge$S

#### an n-by-n general equilibrium model with Cobb-Douglas functions.
f <- function(n, policy = NULL, z0 = rep(100 * n, n), numberOfPeriods = 30,
              Beta = matrix(1 / n, n, n), n.firm = n - 1) {
  ge <- sdm2(
    A = function(state) {
      CD_A(alpha = rep(n, n), Beta = Beta, p = state$p)
    },
    B = {
      tmp <- diag(n)
      tmp[, (n.firm + 1):n] <- 0
      tmp
    },
    S0Exg = {
      tmp <- matrix(NA, n, n)
      for (k in (n.firm + 1):n) tmp[k, k] <- 100 * n
      tmp
    },
    numeraire = n,
    policy = policy,
    z0 = z0,
    maxIteration = 1,
    numberOfPeriods = numberOfPeriods,
    names.agent = c(paste0("firm", 1:n.firm), paste0("consumer", 1:(n - n.firm))),
    ts = TRUE
  )
  print(ge$z)
  print(ge$p)
  invisible(ge)
}

n <- 4
f(n, n.firm = n - 2)
## a spot equilibrium path
ge <- f(n, policy = policyMarketClearingPrice, z0 = runif(n, 10 * n, 100 * n), n.firm = n - 2)
matplot(ge$ts.z, type = "b", pch = 20)
matplot(ge$ts.p, type = "b", pch = 20)
}
}
