% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fe_stand.R
\name{fe_stand}
\alias{fe_stand}
\title{User Friendly Construction of an \strong{fe_stand} Object from a Data Frame}
\usage{
fe_stand(
  x,
  tree_id_col,
  species_id_col,
  time_yr_col,
  dbh_cm_col,
  area_ha,
  stand_id = "my_fe_stand",
  layer_key_col = NA,
  age_yr_col = NA,
  height_m_col = NA,
  crown_base_height_m_col = NA,
  crown_radius_m_col = NA,
  removal_col = NA,
  ingrowth_col = NA,
  n_rep_ha_col = NA,
  small_trees = data.frame(),
  verbose = TRUE
)
}
\arguments{
\item{x}{\code{data.frame} to be coerced into the goal object. Each line of
\code{x} must represent a single tree. As a minimum requirement, there must
be a column for the tree id, the species id, the time (in years), the
diameter at breast height (dbh), each.}

\item{tree_id_col}{name of the column in \code{x} which contains the tree
id's (\code{character}, required, must not contain missing values)}

\item{species_id_col}{name of the column in \code{x} which contains the
species id's. Must be an object of one  of the \code{fe_species} classes
supported by this package. This column is required, must not contain
missing values.}

\item{time_yr_col}{name of the column in \code{x} which provides time
information in years (\code{character}, required, must not contain missing
values)}

\item{dbh_cm_col}{name of the column in \code{x} which contains the dbh in cm
(\code{character}, required, must not contain missing values)}

\item{area_ha}{size of the stand area in ha (\code{numeric}. It is possible
to have no defined stand area; in this case area_ha must be
\code{NULL}, and n_rep_ha_col must be given (with no missing values).}

\item{stand_id}{arbitrary id of the stand (\code{character}, default:
"my_fe_stand")}

\item{layer_key_col}{name of the column in \code{x} that contains codes for
the stand layer a given tree belongs to. These codes are whole numbers.
The following values are allowed: 1 - Main stand, 2 - Understorey,
3 - Pregeneration (ger: "Vorausverjuengung"), 4 - Remnant trees, hold over
trees, veteran trees (ger: "Nachhiebsreste", "Ueberhaelter", "Altbaeume").
Must not contain missing values if provided. If not provided, it will be
set to 1 (main stand) for every tree.}

\item{age_yr_col}{name of the column in \code{x} which provides the tree ages
in years (\code{character}, optional, may contain missing values)}

\item{height_m_col}{name of the column in \code{x} which provides the tree
heights in m (\code{character}, optional, may contain missing values)}

\item{crown_base_height_m_col}{name of the column in \code{x} which provides
the crown base heights in m (\code{character}, optional, may contain
missing values)}

\item{crown_radius_m_col}{name of the column in \code{x} which provides the
crown radii in m (\code{character}, optional, may contain missing values)}

\item{removal_col}{name of the column in \code{x} which provides the tree's
removal status. If \code{TRUE}, this indicates the tree was removed or dead
at the time indicated in \code{age_yr_col} (\code{character}, optional,
must not contain missing values if provided). If not provided, the removal
status will be \code{FALSE} for all trees in the resulting \code{fe_stand}
object.}

\item{ingrowth_col}{name of the column in \code{x} which provides the tree's
ingrowth status. If \code{TRUE}, this indicates a tree that grew newly in
at the time indicated in \code{age_yr_col} (\code{character}, optional,
must not contain missing values if provided). If not provided, the ingrowth
status will be \code{FALSE} for all trees in the resulting \code{fe_stand}
object.}

\item{n_rep_ha_col}{name of the column in the trees data frame which provides
each tree's representation number per ha. Not required if a stand area is
provided under \code{area_ha}. If a stand outline is given, n_rep_ha will
be always recalculated based on the outline and the tree positions.}

\item{small_trees}{An \code{fe_stand} object does contain an extra slot for
small trees, defined as trees which are too small to have an own dbh (i.e.
having a height > 1.3 m). So far, this slot is still experimental. The only
requirement is that it is a \code{data.frame}. Such a \code{data.frame} can
be provided via this parameter, it will be directly put into the goal
object's \code{small_trees} slot. The default is a \code{data.frame} with
zero rows and zero columns (\code{data.frame()}).}

\item{verbose}{\code{logical}, if \code{TRUE} (default) the tree size
variables will be checked for plausible orders of magnitude after
successful construction of the \code{fe_stand} object. In case of a
potential implausibility, a warning will be raised. The purpose of this
mechanism is to avoid unit mismatches.}
}
\value{
If the user input allows to construct a well-defined \code{fe_stand}
object, this object will be returned. If not, the function will terminate
with an error.
}
\description{
\code{fe_stand()} provides a user-friendly interface for the constructor
\code{\link{new_fe_stand}}. While the constructor does not prevent users from
creating malformed \code{fe_stand} objects, \code{fe_stand} does everything
to achieve a well-defined object mostly based on an initial \code{data.frame}
that might be, e.g. drawn out of a user's own data base.
}
\details{
The initial \code{data.frame} (or even nicer, \code{tibble}) provided by the
user must contain a a minimum set of columns (tree id, species id, time
variable, diameter at breast height). These columns must not contain missing
values. Other columns (containing tree height, height to crown base, crown
radius, tree age) are optional for the user to provide. If provided, they may
contain missing values. If not provided these columns will only contain
missing values in the \code{fe_stand} object. The columns about the trees'
removal and ingrowth status are also optional, but if provided, they must
\emph{not} contain missing values. If not provided, both columns will be filled
with \code{FALSE} in the resulting \code{fe_stand} object.

The columns from the user's \code{data.frame} that correspond to the columns
defined in \code{fe_stand} objects will turn up in the object under standard
names. All other columns that might be in the \code{data.frame} will be
transferred to the \code{fe_stand} object with their orignal names. It is the
user's responsibility to take care of them.

\code{fe_stand} will automatically add a column \code{n_rep_ha} which
contains for each tree the number of trees it represents per ha. This may
seem redundant if looking at \code{fe_stand} objects alone, but it allows a
broad range of evaluation functions to be applied to different objects
containing trees.
}
\examples{
# Constructing an fe_stand object based on the minimum required information
# - make data.frame (or, nicer, a tibble) with stand information from
#   scratch
candidate_stand <- tibble::tibble(
  tree_no    = as.character(c(1:100)),
  species_id = as_fe_species_tum_wwk_short(c(rep("1", 30), rep("5", 70))),
  time_yr    = 2022,
  dbh        = c(rnorm(30, 45, 12), rnorm(70, 38, 9))
)

# - call fe_stand
goal_fe_stand_object <- fe_stand(
  x = candidate_stand,
  tree_id_col = "tree_no",
  species_id_col = "species_id",
  time_yr_col = "time_yr",
  dbh_cm_col = "dbh",
  area_ha = 0.33
)


# Using raw data that could come out of a user's data bases; here one
# example stands (spruce_beech_1_raw) provided with the ForestElementsR
# package
spruce_beech_1_raw$year <- 2022 # No time information in the data frame
spruce_beech_1_raw$species <- as_fe_species_tum_wwk_short(
  spruce_beech_1_raw$species
)

spruce_beech_stand <- fe_stand(
  spruce_beech_1_raw,
  tree_id_col = "no",
  species_id_col = "species",
  time_yr_col = "year",
  dbh_cm_col = "d",
  area_ha = 0.49,
  stand_id = spruce_beech_1_raw[1, ]$stand,
  age_yr_col = "age",
  height_m_col = "h",
  crown_base_height_m_col = "hcb",
  crown_radius_m_col = "crad"
)

# Little summary
spruce_beech_stand |> summary()

}
