% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lorenz_curve.r
\name{lorenz_curve}
\alias{lorenz_curve}
\title{Plot a Lorenz curve from regional industrial counts}
\usage{
lorenz_curve(mat, plot = TRUE, pdf = TRUE, pdf_location = NULL)
}
\arguments{
\item{mat}{An incidence matrix with regions in rows and industries in columns. The input can also be a vector of industrial regional count (a matrix with n regions in rows and a single column).}

\item{plot}{Logical; shall the curve be automatically plotted? Defaults to TRUE. If set to TRUE, the function will return x y coordinates that you can latter use to plot and customize the curve.}

\item{pdf}{Logical; shall a pdf be saved?  Defaults to FALSE. If set to TRUE, a pdf with all will be compiled and saved to R's temp dir if no 'pdf_location' is specified.}

\item{pdf_location}{Output location of pdf file}
}
\value{
If `plot = FALSE`, the function returns a list with two components:
  - `cum.reg`: A vector of cumulative proportions of regions.
  - `cum.out`: A vector of cumulative proportions of industrial output.
If `plot = TRUE`, the function generates a plot of the Lorenz curve and does not return a value.
}
\description{
This function plots a Lorenz curve from regional industrial counts. This curve gives an indication of the unequal distribution of an industry accross regions.
}
\examples{
## generate vectors of industrial count
ind <- c(0, 10, 10, 30, 50)

## run the function
lorenz_curve (ind)
lorenz_curve (ind, plot = FALSE)

## generate a region - industry matrix
mat = matrix (
c (0, 1, 0, 0,
0, 1, 0, 0,
0, 1, 0, 0,
0, 1, 0, 1,
0, 1, 1, 1), ncol = 4, byrow = TRUE)
rownames(mat) <- c ("R1", "R2", "R3", "R4", "R5")
colnames(mat) <- c ("I1", "I2", "I3", "I4")

## run the function
lorenz_curve (mat)
lorenz_curve (mat, plot = FALSE)

## run the function by aggregating all industries
lorenz_curve (rowSums(mat))
lorenz_curve (rowSums(mat), plot = FALSE)

## run the function for industry #1 only (perfect equality)
lorenz_curve (mat[,1])
lorenz_curve (mat[,1], plot = FALSE)

## run the function for industry #2 only (perfect equality)
lorenz_curve (mat[,2])
lorenz_curve (mat[,2], plot = FALSE)

## run the function for industry #3 only (perfect unequality)
lorenz_curve (mat[,3])
lorenz_curve (mat[,3], plot = FALSE)

## run the function for industry #4 only (top 40\% produces 100\% of the output)
lorenz_curve (mat[,4])
lorenz_curve (mat[,4], plot = FALSE)

## Compare the distribution of the #industries
oldpar <- par(mfrow = c(2, 2))  # Save the current graphical parameter settings
lorenz_curve (mat[,1])
lorenz_curve (mat[,2])
lorenz_curve (mat[,3])
lorenz_curve (mat[,4])
par(oldpar)  # Reset the graphical parameters to their original values

## Save output as pdf
lorenz_curve (mat, pdf = TRUE)

## To specify an output directory for the pdf,
## specify 'pdf_location', for instance as '/Users/jones/lorenz_curve.pdf'
## lorenz_curve(mat, pdf = TRUE, pdf_location = '/Users/jones/lorenz_curve.pdf')

}
\references{
Lorenz, M. O. (1905) Methods of measuring the concentration of wealth, \emph{Publications of the American Statistical Association} \strong{9}: 209–219
}
\seealso{
\code{\link{hoover_gini}}, \code{\link{locational_gini}}, \code{\link{locational_gini_curve}}, \code{\link{hoover_curve}}, \code{\link{gini}}
}
\author{
Pierre-Alexandre Balland \email{p.balland@uu.nl}
}
\keyword{concentration}
\keyword{inequality}
