% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/window.R
\name{standardize_windows}
\alias{standardize_windows}
\title{Standardize windows of signal data}
\usage{
standardize_windows(
  x,
  standardization_method = c("time_normalize"),
  target_samples = 500,
  target_ms = NULL,
  interpolation_method = c("linear", "spline", "step"),
  align_feature = NULL,
  preserve_amplitude = TRUE,
  preserve_class = FALSE,
  ...
)
}
\arguments{
\item{x}{A \code{windowed} object to standardize}

\item{standardization_method}{A \code{character} string specifying the standardization method.
Currently supported: "time_normalize".}

\item{target_samples}{The desired number of samples for each standardized
window. Default is 500 samples. This parameter takes precedence if both
target_samples and target_ms are provided.}

\item{target_ms}{Alternative specification in milliseconds. If provided and
target_samples is NULL, the function will convert this to samples based on
the signal's sampling frequency.}

\item{interpolation_method}{The method used for interpolation when
resampling. Options are "linear" (default), "spline", or "step".}

\item{align_feature}{Feature to align windows around, either a character
string matching an annotation type or a list of criteria for finding a
specific annotation. Default is NULL (no alignment).}

\item{preserve_amplitude}{Logical. If TRUE (default), maintains original
amplitude range after resampling.}

\item{preserve_class}{Logical. If TRUE, returns a \code{windowed} object with
standardized data frames. If FALSE (default), returns a plain list of data
frames.}

\item{...}{Additional arguments passed to specific standardization methods.}
}
\value{
If \code{preserve_class=TRUE}, a \code{windowed} object containing standardized
data frames. If \code{preserve_class=FALSE}, a plain list of standardized data
frames.
}
\description{
Standardizes \code{windowed} objects by applying various
transformations to each window. This function converts each \code{egm} object in a
\code{windowed} list to a standardized data frame with uniform properties,
facilitating comparison and analysis.
}
\details{
Currently supported standardization methods:
\itemize{
\item \code{time_normalize} - Resamples each window to a standard length by either
dilating or contracting the signal. The result is a signal with a consistent
number of samples regardless of the original window duration.
}

Additional options:
\itemize{
\item \code{align_feature} - If provided, windows will be aligned to center around this
feature (e.g., a specific annotation type like "N" for R-peak). Can be a
character string matching an annotation type or a list of criteria for
annotation matching.
\item \code{preserve_amplitude} - If TRUE (default), maintains the original amplitude
range after resampling. If FALSE, the amplitudes may change due to
interpolation.
}
}
\examples{
\dontrun{
# Read in ECG data
ecg <- read_wfdb("ecg", test_path(), "ecgpuwave")

# Create windows based on sinus rhythm
windows <- window_signal(
  ecg,
  method = "rhythm",
  rhythm_type = "sinus",
  onset_criteria = list(type = "(", number = 0),
  offset_criteria = list(type = ")", number = 2),
  reference_criteria = list(type = "N")
)

# Standardize windows to exactly 500 samples
std_windows <- standardize_windows(
  windows,
  method = "time_normalize",
  target_samples = 500
)

# Alternatively, standardize to 500 milliseconds (depends on sampling frequency)
std_windows_ms <- standardize_windows(
  windows,
  method = "time_normalize",
  target_ms = 500
)

# Standardize windows with QRS alignment
aligned_windows <- standardize_windows(
  windows,
  method = "time_normalize",
  target_samples = 500,
  align_feature = "N"  # Align on QRS complexes
)
}

}
