% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DPI.R
\name{BNs_dag}
\alias{BNs_dag}
\title{Directed acyclic graphs (DAGs) via Bayesian networks (BNs).}
\usage{
BNs_dag(
  data,
  algorithm = c("pc.stable", "hc", "rsmax2"),
  algorithm.args = list(),
  n.boot = 1000,
  seed = NULL,
  strength = 0.85,
  direction = 0.5,
  node.text.size = 1.2,
  edge.width.max = 1.5,
  edge.label.mrg = 0.01,
  file = NULL,
  width = 6,
  height = 4,
  dpi = 500,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{Data.}

\item{algorithm}{\link[bnlearn:structure-learning]{Structure learning algorithms} for building Bayesian networks (BNs). Should be function name(s) from the \code{\link[bnlearn:bnlearn-package]{bnlearn}} package. Better to perform BNs with all three classes of algorithms to check the robustness of results (Briganti et al., 2023).

Defaults to the most common algorithms: \code{"pc.stable"} (PC), \code{"hc"} (HC), and \code{"rsmax2"} (RS), for the three classes, respectively.
\itemize{
\item (1) \link[bnlearn:constraint-based algorithms]{Constraint-based Algorithms}
\itemize{
\item PC:
\code{"\link[bnlearn:pc.stable]{pc.stable}"}
(\emph{the first practical constraint-based causal structure learning algorithm by Peter & Clark})
\item Others:
\code{"\link[bnlearn:gs]{gs}"},
\code{"\link[bnlearn:iamb]{iamb}"},
\code{"\link[bnlearn:fast.iamb]{fast.iamb}"},
\code{"\link[bnlearn:inter.iamb]{inter.iamb}"},
\code{"\link[bnlearn:iamb.fdr]{iamb.fdr}"}
}
\item (2) \link[bnlearn:score-based algorithms]{Score-based Algorithms}
\itemize{
\item Hill-Climbing:
\code{"\link[bnlearn:hc]{hc}"}
(\emph{the hill-climbing greedy search algorithm, exploring DAGs by single-edge additions, removals, and reversals, with random restarts to avoid local optima})
\item Others:
\code{"\link[bnlearn:tabu]{tabu}"}
}
\item (3) \link[bnlearn:hybrid algorithms]{Hybrid Algorithms} (combination of constraint-based and score-based algorithms)
\itemize{
\item Restricted Maximization:
\code{"\link[bnlearn:rsmax2]{rsmax2}"}
(\emph{the general 2-phase restricted maximization algorithm, first restricting the search space and then finding the optimal [maximizing the score of] network structure in the restricted space})
\item Others:
\code{"\link[bnlearn:mmhc]{mmhc}"},
\code{"\link[bnlearn:h2pc]{h2pc}"}
}
}}

\item{algorithm.args}{An optional list of extra arguments passed to the algorithm.}

\item{n.boot}{Number of bootstrap samples (for learning a more "stable" network structure). Defaults to \code{1000}.}

\item{seed}{Random seed for replicable results. Defaults to \code{NULL}.}

\item{strength}{Stability threshold of edge \emph{strength}: the minimum proportion (probability) of BNs (among the \code{n.boot} bootstrap samples) in which each edge appears.
\itemize{
\item Defaults to \code{0.85} (85\%).
\item Two reverse directions share the same edge strength.
\item Empirical frequency (?~100\%) will be mapped onto edge \emph{width/thickness} in the final integrated \code{DAG}, with wider (thicker) edges showing stronger links, though they usually look similar since the default range has been limited to 0.85~1.
}}

\item{direction}{Stability threshold of edge \emph{direction}: the minimum proportion (probability) of BNs (among the \code{n.boot} bootstrap samples) in which a direction of each edge appears.
\itemize{
\item Defaults to \code{0.50} (50\%).
\item The proportions of two reverse directions add up to 100\%.
\item Empirical frequency (?~100\%) will be mapped onto edge \emph{greyscale/transparency} in the final integrated \code{DAG}, with its value shown as edge text label.
}}

\item{node.text.size}{Scalar on the font size of node (variable) labels. Defaults to \code{1.2}.}

\item{edge.width.max}{Maximum value of edge strength to scale all edge widths. Defaults to \code{NULL} (for undirected correlation networks) and \code{1.5} (for directed acyclic networks to better display arrows).}

\item{edge.label.mrg}{Margin of the background box around the edge label. Defaults to \code{0.01}.}

\item{file}{File name of saved plot (\code{".png"} or \code{".pdf"}).}

\item{width, height}{Width and height (in inches) of saved plot. Defaults to \code{6} and \code{4}.}

\item{dpi}{Dots per inch (figure resolution). Defaults to \code{500}.}

\item{verbose}{Print information about BN algorithm and number of bootstrap samples when running the analysis. Defaults to \code{TRUE}.}

\item{...}{Arguments passed on to \code{\link[qgraph:qgraph]{qgraph()}}.}
}
\value{
Return a list (class \code{bns.dag}) of Bayesian network results and \code{\link[qgraph:qgraph]{qgraph}} object.
}
\description{
Directed acyclic graphs (DAGs) via Bayesian networks (BNs). It uses \code{\link[bnlearn:arc.strength]{bnlearn::boot.strength()}} to estimate the strength of each edge as its \emph{empirical frequency} over a set of networks learned from bootstrap samples. It computes (1) the probability of each edge (modulo its direction) and (2) the probabilities of each edge's directions conditional on the edge being present in the graph (in either direction). Stability thresholds are usually set as \code{0.85} for \emph{strength} (i.e., an edge appearing in more than 85\% of BNs bootstrap samples) and \code{0.50} for \emph{direction} (i.e., a direction appearing in more than 50\% of BNs bootstrap samples) (Briganti et al., 2023). Finally, for each chosen algorithm, it returns the stable Bayesian network as the final DAG.
}
\examples{
\donttest{bn = BNs_dag(airquality, seed=1)
bn
# bn$pc.stable
# bn$hc
# bn$rsmax2

## All DAG objects can be directly plotted
## or saved with print(..., file="xxx.png")
# bn$pc.stable$DAG.edge
# bn$pc.stable$DAG.strength
# bn$pc.stable$DAG.direction
# bn$pc.stable$DAG
# ...
}
\dontrun{

print(bn, file="airquality.png")
# will save three plots with auto-modified file names:
- "airquality_BNs.DAG.01_pc.stable.png"
- "airquality_BNs.DAG.02_hc.png"
- "airquality_BNs.DAG.03_rsmax2.png"

# arrange multiple plots using aplot::plot_list()
# install.packages("aplot")
c1 = cor_net(airquality, "cor")
c2 = cor_net(airquality, "pcor")
bn = BNs_dag(airquality, seed=1)
mytheme = theme(plot.title=element_text(hjust=0.5))
p = aplot::plot_list(
  plot(c1),
  plot(c2),
  plot(bn$pc.stable$DAG) + mytheme,
  plot(bn$hc$DAG) + mytheme,
  plot(bn$rsmax2$DAG) + mytheme,
  design="111222
          334455",
  tag_levels="A"
)  # return a patchwork object
ggsave(p, filename="p.png", width=12, height=8, dpi=500)
ggsave(p, filename="p.pdf", width=12, height=8)
}

}
\references{
Briganti, G., Scutari, M., & McNally, R. J. (2023). A tutorial on Bayesian networks for psychopathology researchers. \emph{Psychological Methods, 28}(4), 947--961. \doi{10.1037/met0000479}

Burger, J., Isvoranu, A.-M., Lunansky, G., Haslbeck, J. M. B., Epskamp, S., Hoekstra, R. H. A., Fried, E. I., Borsboom, D., & Blanken, T. F. (2023). Reporting standards for psychological network analyses in cross-sectional data. \emph{Psychological Methods, 28}(4), 806--824. \doi{10.1037/met0000471}

Scutari, M., & Denis, J.-B. (2021). \emph{Bayesian networks: With examples in R} (2nd ed.). Chapman and Hall/CRC. \doi{10.1201/9780429347436}

\url{https://www.bnlearn.com/}
}
\seealso{
\link{S3method.network}

\code{\link[=DPI_dag]{DPI_dag()}}

\code{\link[=cor_net]{cor_net()}}
}
