# ---------- core fit ----------
#'
#' Internal function for multiple DEoptim runs
#' @keywords internal
.fit_mix2_core <- function(x, family = c("lognormal", "normal"),
                           lower = NULL, upper = NULL,
                           NP = 100, itermax = 10000, reltol = 5e-6, steptol = 50,
                           F = 0.8, CR = 0.9, strategy = 2, parallelType = 0,
                           packages = c("stats"), parVar = NULL,
                           n_runs = 20, quiet = 2,
                           par_init = NULL, pgtol = 1e-8) {
  family <- match.arg(family)
  if (!is.numeric(x) || any(is.na(x))) stop("x must be numeric vector without NA")
  if (family == "lognormal" && any(x <= 0)) stop("x must be >0 for lognormal")

  if (family == "lognormal") {
    if (is.null(lower) || is.null(upper)) {
      bb <- default_bounds_lognorm2(x); lower <- lower %||% bb$lower; upper <- upper %||% bb$upper
    }
    loglik_fn <- loglik_lognorm
  } else {
    if (is.null(lower) || is.null(upper)) {
      bb <- default_bounds_norm2(x); lower <- lower %||% bb$lower; upper <- upper %||% bb$upper
    }
    loglik_fn <- loglik_norm
  }

  demulti <- .run_demulti(loglik_fn, x, lower, upper,
                          NP, itermax, reltol, steptol, F, CR, strategy,
                          parallelType, packages, parVar,
                          n_runs, quiet)
  de_bestpar <- demulti$best_fit
  local_from_de <- tryCatch({
    optim(de_bestpar, fn = function(par) -loglik_fn(par, x),
          method = "L-BFGS-B", lower = lower, upper = upper,
          control = list(factr = 1e7, pgtol = pgtol, maxit = 100000))
  }, error = function(e) e)

  local_from_init <- NULL
  if (!is.null(par_init)) {
    local_from_init <- tryCatch({
      optim(par_init, fn = function(par) -loglik_fn(par, x),
            method = "L-BFGS-B", lower = lower, upper = upper,
            control = list(factr = 1e7, pgtol = pgtol, maxit = 100000))
    }, error = function(e) e)
  }

  candidates <- list()
  if (!inherits(local_from_de, "error")) {
    candA_par <- order_components(local_from_de$par)
    candA_loglik <- -local_from_de$value
    candidates[[length(candidates) + 1]] <- list(par = candA_par, logLik = candA_loglik, source = "DEoptim")
  } else {
    fb_par <- order_components(de_bestpar)
    fb_loglik <- loglik_fn(fb_par, x)
    candidates[[length(candidates) + 1]] <- list(par = fb_par, logLik = fb_loglik, source = "DEoptim_nolocal")
  }
  if (!is.null(local_from_init) && !inherits(local_from_init, "error")) {
    candB_par <- order_components(local_from_init$par)
    candB_loglik <- -local_from_init$value
    candidates[[length(candidates) + 1]] <- list(par = candB_par, logLik = candB_loglik, source = "par_init")
  }
  if (length(candidates) == 0) stop("Optimization failed")

  best_idx <- which.max(sapply(candidates, function(z) z$logLik))
  best <- candidates[[best_idx]]
  best_par <- order_components(best$par)
  names(best_par) <- c("p", "m1", "s1", "m2", "s2")
  best_logLik <- best$logLik

  k <- 5L; n <- length(x)
  list(family = family, data = x, par = best_par, logLik = best_logLik,
       AIC = -2 * best_logLik + 2 * k,
       BIC = -2 * best_logLik + log(n) * k,
       convergence = if (!inherits(local_from_de, "error")) local_from_de$convergence else NA,
       message = if (!inherits(local_from_de, "error")) local_from_de$message else "DEoptim fallback")
}
