% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PreEst.glasso.R
\name{PreEst.glasso}
\alias{PreEst.glasso}
\title{Precision Matrix Estimation via Graphical Lasso}
\usage{
PreEst.glasso(X, method = list(type = "fixed", param = 1), parallel = FALSE)
}
\arguments{
\item{X}{an \eqn{(n\times p)} data matrix where each row is an observation.}

\item{method}{a list containing following parameters, \describe{
\item{type}{one of \code{'fixed'},\code{'confidence'}, or \code{'BIC'}.}
\item{param}{either a numeric value or vector of values.}
}}

\item{parallel}{a logical; \code{TRUE} for using half the cores available, \code{FALSE} otherwise.}
}
\value{
a named list containing: \describe{
\item{C}{a \eqn{(p\times p)} estimated precision matrix.}
\item{BIC}{a dataframe containing \eqn{\lambda} values and corresponding BIC scores with \code{type='BIC'} method.}
}
}
\description{
Given a sample covariance matrix \eqn{S}, graphical lasso aims at estimating sparse precision matrix \eqn{X} - inverse
of covariance. It solves a following optimization problem,
\deqn{\textrm{max}_X
\log\textrm{det}X - <S,X> - \lambda \|X \|_1 \textrm{ such that } X \succ 0}
where \eqn{\lambda} a regularization parameter, \eqn{<S,X>=tr(S^T X)} , \eqn{\|X\|_1 = \sum X_{ij}} and \eqn{X\succ 0} indicates positive definiteness. We provide three
modes of computations, \code{'fixed'},\code{'confidence'}, or \code{'BIC'} with respect to \eqn{\lambda}. Please see the section below for more details.
}
\section{regularization parameters}{

We currently provide three options for solving the problem, \code{'fixed'},\code{'confidence'}, or \code{'BIC'} with respect to \eqn{\lambda}.
When the method type is \code{'fixed'}, the parameter should be a single numeric value as a user-defined \eqn{\lambda} value. Likewise,
method type of \code{'confidence'} requires a singule numeric value in \eqn{(0,1)}, where the value is set heuristically
according to
\deqn{
\rho = \frac{t_{n-2}(\gamma) \max S_{ii}S_{jj}}{\sqrt{n-2+ t_{n-2}^2(\gamma)}}
}
for a given confidence level \eqn{\gamma \in (0,1)} as proposed by Banerjee et al. (2006).
Finally, \code{'BIC'} type requires a vector of \eqn{\lambda} values and opts for a lambda value with the lowest BIC values
as proposed by Yuan and Lin (2007).
}

\examples{
\donttest{
## generate data from multivariate normal with Identity precision.
pdim = 10
data = matrix(rnorm(100*pdim), ncol=pdim)

## prepare input arguments for diefferent scenarios
lbdvec <- c(0.01,0.1,1,10,100)              # a vector of regularization parameters
list1 <- list(type="fixed",param=1.0)       # single regularization parameter case
list2 <- list(type="confidence",param=0.95) # single confidence level case
list3 <- list(type="BIC",param=lbdvec)      # multiple regularizers with BIC selection

## compute with different scenarios
out1 <- PreEst.glasso(data, method=list1)
out2 <- PreEst.glasso(data, method=list2)
out3 <- PreEst.glasso(data, method=list3)

## visualize
opar <- par(no.readonly=TRUE)
par(mfrow=c(2,2), pty="s")
image(diag(pdim)[,pdim:1], main="Original Precision")
image(out1$C[,pdim:1],     main="glasso::lambda=1.0")
image(out2$C[,pdim:1],     main="glasso::Confidence=0.95")
image(out3$C[,pdim:1],     main="glasso::BIC selection")
par(opar)
}

}
\references{
\insertRef{banerjee_convex_2006}{CovTools}

\insertRef{yuan_model_2007}{CovTools}

\insertRef{friedman_sparse_2008}{CovTools}
}
