% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/perform_regression_test.R
\name{perform_regression_test}
\alias{perform_regression_test}
\title{Perform a test on the slope coefficient of a univariate linear regression}
\usage{
perform_regression_test(
  X,
  Y,
  nBootstrap = 100,
  show_progress = TRUE,
  bootstrapOptions = NULL
)
}
\arguments{
\item{X}{numeric univariate input vector resembling the independent variables}

\item{Y}{numeric univariate input vector the dependent variables}

\item{nBootstrap}{numeric value of the amount of bootstrap resamples}

\item{show_progress}{logical value indicating whether to show a progress bar}

\item{bootstrapOptions}{This can be one of \itemize{
  \item \code{NULL} This uses the default options \code{type_boot = "indep"},
  \code{type_stat = "eq"}.

  \item a list with at most 2 elements names \itemize{
        \item \code{type_boot} type of bootstrap resampling scheme. It must be
        one of
        \itemize{
           \item \code{"indep"} for the independence bootstrap
           (i.e. under the null). This is the default.
           \item \code{"NP"} for the non-parametric bootstrap
           (i.e. n out of n bootstrap).
           \item \code{"res_bs"} for the residual bootstrap.
           \item \code{"hybrid_null_bs"} for the hybrid null bootstrap
           \item \code{"fixed_design_bs"} for the fixed design bootstrap
           \item \code{"fixed_design_bs_Hnull"} for the fixed design null bootstrap.
        }

        \item \code{type_stat} type of test statistic to be used.  It must be
        one of
        \itemize{
          \item \code{"eq"} for the equivalent test statistic
          \eqn{ T_n^* = \sqrt{n} | \hat{b}^*  |  }.
          This is the default.

          \item \code{"cent"} for the centered test statistic
          \eqn{ T_n^* = \sqrt{n} | \hat{b}^* - \hat{b} |  }
        }
        For each \code{type_boot} there is only one valid choice of \code{type_stat}
        to be made. If \code{type_stat} is not specified, the valid choice is
        automatically used.

  }
  \item \code{"all"} this gives test results for all theoretically valid
  combinations of bootstrap resampling schemes.

  \item \code{"all and also invalid"} this gives test results for all possible
  combinations of bootstrap resampling schemes and test statistics, including
  invalid ones.
}
A warning is raised if the given combination of \code{type_boot} and
\code{type_stat} is theoretically invalid.}
}
\value{
A class object with components \itemize{
   \item \code{pvals_df} a dataframe of p-values and bootstrapped test statistics:

   These are the p-values for the combinations of bootstrap resampling schemes,
   test statistics (centered and equivalent).

   It also contains the vectors of bootstrap test statistics
   for each of the combinations.

   \item \code{true_stat} a named vector of size 1 containing the true test
   statistic.

   \item \code{nBootstrap} Number of bootstrap repetitions.

   \item \code{data} named list of the used input data, i.e. X and Y.

   \item \code{nameMethod} string for the name of the method used.

   \item \code{beta} numeric value of the estimated slope of the regression model.
}
}
\description{
This function performs a bootstrap regression test for given data X,Y.
The null hypothesis corresponds of a slope coefficient of zero, versus the
alternative hypothesis of a non-zero slope coefficient.
It uses an independence/null bootstrap \code{"indep"}, a non-parametric \code{"NP"},
a residual bootstrap \code{"res_bs"}, a fixed design bootstrap \code{"fixed_design_bs"},
a fixed design null bootstrap \code{"fixed_design_bs_Hnull"}, a hybrid null
bootstrap \code{"hybrid_null_bs"} as bootstrap resampling schemes to perform
the bootstrap. This function gives the corresponding p-values, the true test
statistic and the bootstrap-version test statistics. Furthermore, it also
gives the estimated slope.The default (and valid) method implemented
in this function is the null bootstrap, together with the equivalent test
statistic. Via the \code{bootstrapOptions} argument, the user can specify other
bootstrap resampling schemes and test statistics.
}
\examples{
n <- 100

# Under H1
X_data <- rnorm(n)
Y_data <-  X_data + rnorm(n)   #Y = X + epsilon
result <- perform_regression_test(X_data, Y_data, nBootstrap = 100,
                        bootstrapOptions =  list(type_boot = "indep",
                                                 type_stat = "eq"))
print(result)
plot(result)

# Under H0
X_data <- rnorm(n)
Y_data <-  0 * X_data + rnorm(n)   # (as b = 0 under H0)
result <- perform_regression_test(X_data, Y_data, nBootstrap = 100)
print(result)
plot(result)

}
\references{
Derumigny, A., Galanis, M., Schipper, W., & van der Vaart, A. (2025).
Bootstrapping not under the null?
ArXiv preprint, \doi{10.48550/arXiv.2512.10546}
}
\seealso{
\code{\link{perform_GoF_test},\link{perform_independence_test}}.
The print and plot methods, such as \code{\link{plot.bootstrapTest}}.
}
