% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdpseudoinv}
\alias{bdpseudoinv}
\title{Compute Matrix Pseudoinverse (In-Memory)}
\usage{
bdpseudoinv(X, threads = NULL)
}
\arguments{
\item{X}{Numeric matrix or vector to be pseudoinverted.}

\item{threads}{Optional integer. Number of threads for parallel computation.
If NULL, uses maximum available threads.}
}
\value{
The pseudoinverse matrix of X.
}
\description{
Computes the Moore-Penrose pseudoinverse of a matrix using SVD decomposition.
This implementation handles both square and rectangular matrices, and provides
numerically stable results even for singular or near-singular matrices.
}
\details{
The Moore-Penrose pseudoinverse (denoted A+) of a matrix A is computed using
Singular Value Decomposition (SVD).

For a matrix A = U\emph{Sigma}V^T (where ^T denotes transpose), the pseudoinverse is
computed as:

\deqn{A^+ = V \Sigma^+ U^T}

where Sigma+ is obtained by taking the reciprocal of non-zero singular values.
}
\section{Mathematical Details}{

\itemize{
\item SVD decomposition: \eqn{A = U \Sigma V^T}
\item Pseudoinverse: \eqn{A^+ = V \Sigma^+ U^T}
\item \eqn{\Sigma^+_{ii} = 1/\Sigma_{ii}} if \eqn{\Sigma_{ii} > \text{tolerance}}
\item \eqn{\Sigma^+_{ii} = 0} otherwise
}

Key features:
\itemize{
\item Robust computation:
\itemize{
\item Handles singular and near-singular matrices
\item Automatic threshold for small singular values
\item Numerically stable implementation
}
\item Implementation details:
\itemize{
\item Uses efficient SVD algorithms
\item Parallel processing support
\item Memory-efficient computation
\item Handles both dense and sparse inputs
}
}

The pseudoinverse satisfies the Moore-Penrose conditions:
\itemize{
\item \eqn{AA^+A = A}
\item \eqn{A^+AA^+ = A^+}
\item \eqn{(AA^+)^* = AA^+}
\item \eqn{(A^+A)^* = A^+A}
}
}

\examples{
library(BigDataStatMeth)

# Create a singular matrix
X <- matrix(c(1,2,3,2,4,6), 2, 3)  # rank-deficient matrix

# Compute pseudoinverse
X_pinv <- bdpseudoinv(X)

# Verify Moore-Penrose conditions
# 1. X \%*\% X_pinv \%*\% X = X
all.equal(X \%*\% X_pinv \%*\% X, X)

# 2. X_pinv \%*\% X \%*\% X_pinv = X_pinv
all.equal(X_pinv \%*\% X \%*\% X_pinv, X_pinv)

}
\references{
\itemize{
\item Golub, G. H., & Van Loan, C. F. (2013). Matrix Computations, 4th Edition.
Johns Hopkins University Press.
\item Ben-Israel, A., & Greville, T. N. E. (2003). Generalized Inverses:
Theory and Applications, 2nd Edition. Springer.
}
}
\seealso{
\itemize{
\item \code{\link{bdpseudoinv_hdf5}} for HDF5-stored matrices
\item \code{\link{bdSVD_hdf5}} for singular value decomposition
}
}
