#' Posterior for alpha0
#'
#' @param alpha0 A scalar defining the parameter for the Dirichlet process prior
#'    that controls the number of clusters (or its initial values)
#' @param a The hyperparameter value for the shape parameter in the gamma prior for alpha0
#' @param b The hyperparameter value for the scale parameter in the gamma prior for alpha0
#' @param N A scalar representing the number of data sequences
#' @param cluster A vector containing the cluster assignments for the data sequences (or its initial values)
#'
#' @returns A numerical value corresponding to a sample from the posterior of alpha0
#'
#' @note
#' This function is called within the Gibbs sampler, but it can be called seperately.
#'
#' @examples
#' postalpha0(alpha0 = 1/100, a = 2, b = 1000, N = 5, cluster = c(1,1,2,1,1))
#'
#' @export
#'
postalpha0 <- function(alpha0, a, b, N, cluster){

  d <- length(unique(cluster))

  kappa <- stats::rbeta(1, alpha0 + 1, N)

  pik <- (a + d - 1)/(N*((1/b) - log(kappa)))

  if(pik > 1){
    pik = 1
  }

  x <- stats::rbinom(1, 1, prob = pik)

  if(x == 0){
    alpha0n <- stats::rgamma(1, a + d - 1, scale = ((1/b) - log(kappa))^(-1))
  } else{
    alpha0n <- stats::rgamma(1, a + d, scale = ((1/b) - log(kappa))^(-1))
  }

  return(alpha0n)
}
