#' @title
#' Update APackOfTheClones objects to the latest format
#'
#' @description
#' This generic function updates objects created by APackOfTheClones to the
#' latest format or structure required by the package. It supports Seurat
#' objects and ggplot objects generated by APackOfTheClones functions.
#'
#' The current possible changes made by the function are:
#' - when called on ggplots generated by [vizAPOTC()] / [APOTCPlot()], layer
#' names will be made unique.
#'
#' @param x An object to update. Supported types are Seurat objects and ggplot
#' objects generated by APackOfTheClones.
#' @param verbose Logical. Whether to print messages about the update process.
#' Default is TRUE.
#' @param ... Additional arguments passed to methods. Currently unused.
#'
#' @return The updated object, or the original object if no update is required.
#' @export
#' @examples
#' data("combined_pbmc")
#' apotc_plot <- vizAPOTC(combined_pbmc)
#' apotc_plot <- updateApotc(apotc_plot, verbose = TRUE)
#' apotc_plot
#'
updateApotc <- function(x, verbose = TRUE, ...) {
    UseMethod("updateApotc")
}

#' @export
updateApotc.Seurat <- function(x, verbose = TRUE, ...) {
    if (verbose) message("No updates required")
    x
}

#' @export
updateApotc.ggplot <- function(x, verbose = TRUE, ...) {

    if (!is_an_apotc_ggplot(x)) {
        if (verbose) warning(
            "input doesn't appear to be a ggplot generated by any ",
            "APackOfTheClones function, or was created prior to v1. ",
            "returning original input."
        )
        return(x)
    }

    if (!is.null(names(x$layers)) && any(duplicated(names(x$layers)))) {
        message(
            "Detected ggplot with duplicated layer names (likely generated before/on APackOfTheClones 1.2.4). ",
            "Updating ggplot layer names by deduplicating."
        )
        names(x$layers) %<>% make.unique()
    }

    x

}
