% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/admm.enet.R
\name{admm.enet}
\alias{admm.enet}
\title{Elastic Net Regularization}
\usage{
admm.enet(
  A,
  b,
  lambda1 = 1,
  lambda2 = 1,
  rho = 1,
  abstol = 1e-04,
  reltol = 0.01,
  maxiter = 1000
)
}
\arguments{
\item{A}{an \eqn{(m\times n)} regressor matrix}

\item{b}{a length-\eqn{m} response vector}

\item{lambda1}{a regularization parameter for \eqn{\ell_1} term}

\item{lambda2}{a regularization parameter for \eqn{\ell_2} term}

\item{rho}{an augmented Lagrangian parameter}

\item{abstol}{absolute tolerance stopping criterion}

\item{reltol}{relative tolerance stopping criterion}

\item{maxiter}{maximum number of iterations}
}
\value{
a named list containing \describe{
\item{x}{a length-\eqn{n} solution vector}
\item{history}{dataframe recording iteration numerics. See the section for more details.}
}
}
\description{
Elastic Net regularization is a combination of \eqn{\ell_2} stability and
\eqn{\ell_1} sparsity constraint simulatenously solving the following,
\deqn{\textrm{min}_x ~ \frac{1}{2}\|Ax-b\|_2^2 + \lambda_1 \|x\|_1 + \lambda_2 \|x\|_2^2}
with nonnegative constraints \eqn{\lambda_1} and \eqn{\lambda_2}. Note that if both lambda values are 0,
it reduces to least-squares solution.
}
\section{Iteration History}{

When you run the algorithm, output returns not only the solution, but also the iteration history recording
following fields over iterates,
\describe{
\item{objval}{object (cost) function value}
\item{r_norm}{norm of primal residual}
\item{s_norm}{norm of dual residual}
\item{eps_pri}{feasibility tolerance for primal feasibility condition}
\item{eps_dual}{feasibility tolerance for dual feasibility condition}
}
In accordance with the paper, iteration stops when both \code{r_norm} and \code{s_norm} values
become smaller than \code{eps_pri} and \code{eps_dual}, respectively.
}

\examples{
## generate underdetermined design matrix
m = 50
n = 100
p = 0.1   # percentange of non-zero elements

x0 = matrix(Matrix::rsparsematrix(n,1,p))
A  = matrix(rnorm(m*n),nrow=m)
for (i in 1:ncol(A)){
  A[,i] = A[,i]/sqrt(sum(A[,i]*A[,i]))
}
b = A\%*\%x0 + sqrt(0.001)*matrix(rnorm(m))

## run example with both regularization values = 1
output = admm.enet(A, b, lambda1=1, lambda2=1)
niter  = length(output$history$s_norm)
history = output$history

## report convergence plot
opar <- par(no.readonly=TRUE)
par(mfrow=c(1,3))
plot(1:niter, history$objval, "b", main="cost function")
plot(1:niter, history$r_norm, "b", main="primal residual")
plot(1:niter, history$s_norm, "b", main="dual residual")
par(opar)

}
\references{
\insertRef{zou_regularization_2005a}{ADMM}
}
\seealso{
\code{\link{admm.lasso}}
}
\author{
Xiaozhi Zhu
}
