#' Plot DBE vs Oxygen Atoms (cf. Herzsprung et al. 2014) with Option for Interactive Plot
#'
#' This function generates a scatter plot of Double Bond Equivalent (DBE) versus the number of oxygen atoms (`o`).
#' It allows for optional customization of colors based on a specified variable (`z_var`) and offers the
#' option to convert the plot to an interactive plotly object.
#'
#' @inheritParams main_docu
#' @inheritDotParams f_colorz
#' @param df A data frame containing the data. The columns `16O` (number of oxygen atoms), `dbe` (DBE values),
#'           and the column specified in `z_var` should be present in the data.
#'
#' @return A ggplot object or a plotly object depending on the `plotly` argument.
# @importFrom ggplot2 ggplot geom_point scale_color_gradientn labs theme_minimal theme annotation_custom
#' @import ggplot2 data.table
#' @importFrom plotly ggplotly
#'
# @examples
# uplot_dbe_vs_o(df = mf_data_demo, z_var = "ri", plotly = TRUE)

uplot_dbe_vs_o <- function(df,
                             z_var = "norm_int",
                             palname = "redblue",
                             col_bar = TRUE,
                             tf = FALSE,
                             logo = TRUE,
                             cex.axis = 12,  # Size of axis text
                             cex.lab = 15,   # Size of axis label
                             plotly = FALSE,  # Option to return plotly object
                             ...
)
{
  # Check if required columns are present
  if (!all(c("16O", "dbe", z_var) %in% names(df))) {
    stop("Error: '16O', 'dbe', or the specified 'z_var' column not found in the data frame.")
  }

  zz <- NULL

  # Prepare data for plotting
  df_dbe_vs_o <- df[, .(`16O`, dbe, zz = get(z_var))]
  setorder(df_dbe_vs_o, zz)

  # Apply color scheme based on the z_var values
  z_col <- f_colorz(df_dbe_vs_o$zz, ...)

  # Create the basic ggplot
  p <- ggplot(df_dbe_vs_o, aes(x = `16O`, y = dbe, color = zz)) +
    geom_point(size = 2, alpha = 0.8) +
    scale_color_gradientn(colors = f_colorz(z_col)) +
    labs(x = "Oxygen atoms", y = "DBE", title = "DBE vs Oxygen Atoms") +
    theme_minimal() +
    theme(
      axis.text = element_text(size = cex.axis),
      axis.title = element_text(size = cex.lab),
      plot.title = element_text(size = cex.lab + 1),
      legend.position = "none"
    )

  # Convert to plotly if requested
  if (plotly) {
    return(plotly::ggplotly(p))
  } else {
    return(p)
  }
}
