% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sentrip.R
\name{sentrip}
\alias{sentrip}
\title{Sensitivity analysis for a triples match in an observational study}
\usage{
sentrip(scores, treated1, gamma = 1, alternative = "greater")
}
\arguments{
\item{scores}{A matrix of scores. Rows correspond to matched triples and
the three columns correspond to the three units in the match. The first unit
is the one treated unit if `treated1 == TRUE` or the one control unit if
`treated1 == FALSE`. The other two columns contain the remaining two units in
the match. These are control units if `treated1 == TRUE` or treated units if
`treated1 == FALSE`. This can easily be created from the triples match using
the [formattrip()] function with `type == "wide"`}

\item{treated1}{A logical vector with length equal to the number of triples.
`TRUE` if there is only one treated unit in the matched triple; `FALSE` if
there are two treated units and only one control unit.
This can easily be created from the triples match using
the [formattrip()] function with `type == "wide"`}

\item{gamma}{The sensitivity parameter \eqn{\Gamma \geq 1}. Setting \eqn{\Gamma = 1}
performs a randomization test that assumes ignorable treatment assignment
given the matched triples}

\item{alternative}{One of `greater`, `less` or `both`. `greater` implies the
alternative hypothesis is that the treatment has a positive effect on the scores,
`less` implies the alternative hypothesis is that the treatment has a negative
effect on the scores, and `both` conducts a two-sided hypothesis test}
}
\value{
Named list with 5 elements: `pval` is the upper bound on the one or two-sided
P-value depending on `alternative`, `deviate` is the deviate that was compared to the
Normal distribution to produce pval, `statistic` is the value of the statistic
that is the sum of scores among treated units, `expectation` is the maximum
expectation of this statistic for the given \eqn{\Gamma}, and `variance` is the
maximum variance of this statistic among treatment assignments that achieve the maximum
expectation
}
\description{
This function parallels [sensitivityfull::senfm()] for full matches.
However, this function does not force the scores used for the test to be
Huber's M-statistic. Instead, scores should be calculated ahead of time
and then entered here. Performs either a randomization test or the
corresponding Rosenbaum sensitivity analysis.
}
\examples{
# Generate some data
set.seed(246)
n <- 30
x <- rnorm(n, 0, 3)
nt <- floor(n * 0.5)
nc <- n - nt
z <- c(rep(1, nt), rep(0, nc))
# Create a distance matrix, everything in one stratum
dist <- dist_mahal(data.frame(x = x), z, rep(1, n))[[1]]
# Create the triples match
triplesm <- triples_st(cost = dist, z = z, solver = "rlemon")
# Create an outcome
y <- 1:40
# Give the outcome some random unit names
names(y) <- sample(1:40)
# Reformat the triples match
ywide <- formattrip(m = triplesm, y = y, type = "wide")
# Turn the outcome into scores, in this case aberrant ranks
ab <- aberrantscores(ywide$ymat, 15, cutoff_dir = "less", tau = 0, treated1 = NULL)
# Conduct a one-sided hypothesis test with a bias of gamma = 1.25
sentrip(scores = ab, treated1 = ywide$treated1, gamma = 1.25, alternative = "greater")
}
\seealso{
sensitivityfull::senfm for more details, especially for the interpretation
of the `expectation` and `variance` components of the output and relevant references.

formattrip for easily creating inputs to this function.
}
