
info <- function(x, mapping = NULL) {

  if((!is.character(x))|length(x)>1){stop("The parameter x must be a character vector of length 1.")}

  if(is.null(mapping)){
    path  <- system.file("extdata", "etcsl_mapping.txt", package = "sumer")
    mapping <- read.csv2(path, sep=";", na.strings="")
    }

  temp       <- split_sumerian(x)
  signs      <- temp$signs
  separators <- temp$separators
  types      <- temp$types

  signs <- standardize_signs(signs)

  n   <- length(signs)

  cuneiform <- "[\\x{12000}-\\x{12500}\u2205]"

  syllables    <- signs
  names        <- signs
  symbols      <- ifelse(str_detect(signs, cuneiform) & str_length(signs) == 1, signs, str_c("\u27E8", signs, "\u27E9"))
  readings     <- signs
  alternatives <- signs

  syllable_vector  <- paste0(",",str_replace_all(mapping$syllables, " ",""), ",")
  default_syllable <- str_remove(mapping$syllables, ",.*")
  default_syllable[is.na(default_syllable)] <- mapping$name[is.na(default_syllable)]

  for(i in seq_along(signs)){
    k <- switch(
      types[i],
      which(str_detect(syllable_vector, paste0(",", signs[i], ","))),
      which(mapping$name      == signs[i]),
      which(mapping$cuneiform == signs[i])
    )

    if(length(k)>0){
      k <- k[1]
      readings[i]     <- ifelse(types[i]==1, signs[i], default_syllable[k])
      names[i]        <- mapping$name[k]
      symbols[i]      <- mapping$cuneiform[k]
      alternatives[i] <- mapping$syllables[k]
    }
  }

  result <- data.frame(
    reading     = readings,
    sign        = symbols,
    name        = names,
    alternatives = alternatives,
    stringsAsFactors = FALSE
  )

  attr(result, "separators") <- separators

  class(result) <- c("info", "data.frame")

  return(result)
}


#\u00a4  operator: of
#\u2218  operator: chaining

"flatten_cuneiform" <- function(x){
  x <- str_replace_all(x, "[()\\[\\]{}\\.,\\-<>\u00a4\u2218]", "-")
  x <- str_replace_all(x, fixed("--"), "-")
  x <- str_replace_all(x, fixed("--"), "-")
  x <- str_replace_all(x, fixed("--"), "-")
  x <- str_replace_all(x, fixed(" -"), " ")
  x <- str_replace_all(x, fixed("- "), " ")
  if(x[1]=="-"){
    x[1] <- ""
  }
  if(x[length(x)]=="-"){
    x[length(x)] <- ""
  }
  return(x)
}


#' @exportS3Method print info

"print.info" <- function(x, flatten=FALSE, ...){

  separators <- attr(x, "separators")

  if(flatten){
    separators <- flatten_cuneiform(separators)
    }
  cuneiform_text <- paste0(c("",x$sign), separators, collapse="")
  cuneiform_text <- str_replace_all(cuneiform_text, "-", "")
  cuneiform_text <- str_replace_all(cuneiform_text, fixed("."), "")

  sign_name_text <- paste0(c("",x$name), separators, collapse="")
  sign_name_text <- str_replace_all(sign_name_text, "-", ".")

  syllable_text <- paste0(c("",x$reading), separators, collapse="")
  syllable_text <- str_replace_all(syllable_text, "\\.(?!\\s|$)", "-")

  for(i in 1:nrow(x)){
    cat(paste0(x$sign[i],"\t",x$name[i],"\t",x$alternatives[i]),"\n")
  }
  cat("\n")

  cat(paste0("syllables      : ", syllable_text, "\n"))
  cat(paste0("sign names     : ", sign_name_text, "\n"))
  cat(paste0("cuneiform text : ", cuneiform_text, "\n"))
}
