% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/general_glm_main_function.R
\name{ssp.glm}
\alias{ssp.glm}
\title{Optimal Subsampling Methods for Generalized Linear Models}
\usage{
ssp.glm(
  formula,
  data,
  subset = NULL,
  n.plt,
  n.ssp,
  family = "binomial",
  criterion = "optL",
  sampling.method = "poisson",
  likelihood = "weighted",
  control = list(...),
  contrasts = NULL,
  ...
)
}
\arguments{
\item{formula}{A model formula object of class "formula" that describes the model to be fitted.}

\item{data}{A data frame containing the variables in the model. Denote \eqn{N} as the number of observations in \code{data}.}

\item{subset}{An optional vector specifying a subset of observations from \code{data} to use for the analysis. This subset will be viewed as the full data.}

\item{n.plt}{The pilot subsample size (first-step subsample size).
This subsample is used to compute the pilot estimator and estimate the optimal subsampling probabilities.}

\item{n.ssp}{The expected size of the optimal subsample (second-step subsample). For \code{sampling.method = 'withReplacement'}, The exact subsample size is \code{n.ssp}. For \code{sampling.method = 'poisson'}, \code{n.ssp} is the expected subsample size.}

\item{family}{\code{family} can be a character string naming a family function, a family function or the result of a call to a family function.}

\item{criterion}{The choices include \code{optA}, \code{optL}(default), \code{LCC} and \code{uniform.}
\itemize{
\item \code{optA} Minimizes the trace of the asymptotic covariance matrix of the subsample estimator.
\item \code{optL} Minimizes the trace of a transformation of the asymptotic covariance matrix. The computational complexity of
optA is \eqn{O(N d^2)} while that of optL is \eqn{O(N d)}.
\item \code{LCC} Local Case-Control sampling probability, used as a baseline subsampling strategy.
\item \code{uniform} Assigns equal subsampling probability
\eqn{\frac{1}{N}} to each observation, serving as a baseline subsampling strategy.
}}

\item{sampling.method}{The sampling method to use. Options include \code{withReplacement}
and \code{poisson} (default). \code{withReplacement} draws exactly \code{n.ssp}
subsamples from size \eqn{N} full dataset with replacement, using the specified
subsampling probabilities. \code{poisson} draws observations independently by
comparing each subsampling probability with a realization of uniform random
variable  \eqn{U(0,1)}.

Differences between methods:
\itemize{
\item Sample size: \code{withReplacement} draws exactly  \code{n.ssp} subsamples while \code{poisson} draws
subsamples with expected size \code{n.ssp}, meaning the actual size may vary.
\item Memory usage: \code{withReplacement} requires the entire dataset to be loaded at once, while \code{poisson}
allows for processing observations sequentially (will be implemented in future version).
\item Estimator performance: Theoretical results show that the \code{poisson} tends to get a
subsample estimator with lower asymptotic variance compared to the
\code{withReplacement}
}}

\item{likelihood}{The likelihood function to use. Options include \code{weighted} (default) and
\code{logOddsCorrection}. A bias-correction likelihood function is required for subsample since unequal subsampling probabilities introduce bias.
\itemize{
\item \code{weighted} Applies a weighted likelihood function where each observation is weighted by the inverse of its subsampling probability.
\item \code{logOddsCorrection} This lieklihood is available only for logistic regression model (i.e., when family is binomial or quasibinomial). It uses a conditional likelihood, where each element of the likelihood represents the probability of \eqn{Y=1}, given that this subsample was drawn.
}}

\item{control}{The argument \code{control} contains two tuning parameters \code{alpha} and \code{b}.
\itemize{
\item \code{alpha} \eqn{\in [0,1]} is the mixture weight of the user-assigned subsampling
probability and uniform subsampling probability. The actual subsample
probability is \eqn{\pi = (1-\alpha)\pi^{opt} + \alpha \pi^{uni}}. This protects the estimator from extreme small
subsampling probability. The default value is 0.
\item \code{b} is a positive number which is used to constaint the poisson subsampling probability. \code{b} close to 0 results in subsampling probabilities closer to uniform probability \eqn{\frac{1}{N}}. \code{b=2} is the default value. See relevant references for further details.
}}

\item{contrasts}{An optional list. It specifies how categorical variables are represented in the design matrix. For example, \code{contrasts = list(v1 = 'contr.treatment', v2 = 'contr.sum')}.}

\item{...}{A list of parameters which will be passed to \code{svyglm()}.}
}
\value{
\code{ssp.glm} returns an object of class "ssp.glm" containing the following components (some are optional):

\describe{
\item{model.call}{The original function call.}
\item{coef.plt}{The pilot estimator. See Details for more information.}
\item{coef.ssp}{The estimator obtained from the optimal subsample.}
\item{coef}{The weighted linear combination of \code{coef.plt} and \code{coef.ssp}. The combination weights depend on the relative size of \code{n.plt} and \code{n.ssp} and the estimated covariance matrices of \code{coef.plt} and \code{coef.ssp.} We blend the pilot subsample information into optimal subsample estimator since the pilot subsample has already been drawn. The coefficients and standard errors reported by summary are \code{coef} and the square root of \code{diag(cov)}.}
\item{cov.ssp}{The covariance matrix of \code{coef.ssp}.}
\item{cov}{The covariance matrix of \code{coef}.}
\item{index.plt}{Row indices of pilot subsample in the full dataset.}
\item{index.ssp}{Row indices of of optimal subsample in the full dataset.}
\item{N}{The number of observations in the full dataset.}
\item{subsample.size.expect}{The expected subsample size, equals to \code{n.ssp} for \code{ssp.glm.} Note that for other functions, such as \link{ssp.relogit}, this value may differ.}
\item{terms}{The terms object for the fitted model.}
}
}
\description{
Draw subsample from full dataset and fit a generalized linear model (GLM) on the subsample. For a quick start, refer to the \href{https://dqksnow.github.io/subsampling/articles/ssp-logit.html}{vignette}.
}
\details{
A pilot estimator for the unknown parameter  \eqn{\beta} is required because both optA and
optL subsampling probabilities depend on \eqn{\beta}. There is no "free lunch" when determining optimal subsampling probabilities. Fortunately the
pilot estimator only needs to satisfy mild conditions. For logistic regression, this
is achieved by drawing a size \code{n.plt} subsample with replacement from full
dataset. The case-control subsample probability is applied, that is, \eqn{\pi_i =
  \frac{1}{2N_1}} for  \eqn{Y_i=1} and  \eqn{\pi_i = \frac{1}{2N_0}} for  \eqn{Y_i=0},
\eqn{i=1,...,N}, where\eqn{N_0} and \eqn{N_1} are the counts of observations with \eqn{Y = 0} and \eqn{Y = 1}, respectively. For other
families, uniform subsampling probabilities are applied. Typically, \code{n.plt} is
relatively small compared to \code{n.ssp}.

When \code{criterion = 'uniform'}, there is no need to compute the pilot estimator. In this case, a size \code{n.plt + n.ssp} subsample will be drawn with uniform sampling probability and \code{coef} is the corresponding  estimator.

As suggested by \code{survey::svyglm()}, for binomial and poisson families, use \code{family=quasibinomial()} and \code{family=quasipoisson()} to avoid a warning "In eval(family$initialize) : non-integer #successes in a binomial glm!". The quasi versions of the family objects give the same point estimates and suppress the warning. Since subsampling methods only rely on point estimates from svyglm() for further computation, using the quasi families does not introduce any issues.

For Gamma family, \code{ssp.glm} returns only the estimation of coefficients, as the dispersion parameter is not estimated.
}
\examples{
# logistic regression
set.seed(2)
N <- 1e4
beta0 <- rep(-0.5, 7)
d <- length(beta0) - 1
corr <- 0.5
sigmax  <- matrix(corr, d, d) + diag(1-corr, d)
X <- MASS::mvrnorm(N, rep(0, d), sigmax)
Y <- rbinom(N, 1, 1 - 1 / (1 + exp(beta0[1] + X \%*\% beta0[-1])))
data <- as.data.frame(cbind(Y, X))
formula <- Y ~ .
n.plt <- 500
n.ssp <- 1000
subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'quasibinomial',
criterion = "optL",
sampling.method = 'poisson',
likelihood = "logOddsCorrection")
summary(subsampling.results)
subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'quasibinomial', 
criterion = "optL",
sampling.method = 'withReplacement', 
likelihood = "weighted")
summary(subsampling.results)
Uni.subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'quasibinomial', 
criterion = 'uniform')
summary(Uni.subsampling.results)
####################
# poisson regression
set.seed(1)
N <-  1e4
beta0 <- rep(0.5, 7)
d <- length(beta0) - 1
X <- matrix(runif(N * d), N, d)
epsilon <- runif(N)
lambda <- exp(beta0[1] + X \%*\% beta0[-1])
Y <- rpois(N, lambda)
data <- as.data.frame(cbind(Y, X))
formula <- Y ~ .
n.plt <- 200
n.ssp <- 600
subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'poisson',
criterion = "optL", 
sampling.method = 'poisson',
likelihood = "weighted")
summary(subsampling.results)
subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'poisson', 
criterion = "optL", 
sampling.method = 'withReplacement',
likelihood = "weighted")
summary(subsampling.results)
Uni.subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'poisson', 
criterion = 'uniform')
summary(Uni.subsampling.results)
##################
# gamma regression
set.seed(1)
N <- 1e4
p <- 3
beta0 <- rep(0.5, p + 1)
d <- length(beta0) - 1
shape <- 2
X <- matrix(runif(N * d), N, d)
link_function <- function(X, beta0) 1 / (beta0[1] + X \%*\% beta0[-1])
scale <- link_function(X, beta0) / shape
Y <- rgamma(N, shape = shape, scale = scale)
data <- as.data.frame(cbind(Y, X))
formula <- Y ~ .
n.plt <- 200
n.ssp <- 1000
subsampling.results <- ssp.glm(formula = formula, 
data = data, 
n.plt = n.plt,
n.ssp = n.ssp,
family = 'Gamma',
criterion = "optL", 
sampling.method = 'poisson',
likelihood = "weighted")
summary(subsampling.results)
}
\references{
Wang, H. (2019). More efficient estimation for logistic regression with optimal subsamples. \emph{Journal of machine learning research}, \strong{20}(132), 1-59.

Ai, M., Yu, J., Zhang, H., & Wang, H. (2021). Optimal subsampling algorithms for big data regressions. \emph{Statistica Sinica}, \strong{31}(2), 749-772.

Wang, H., & Kim, J. K. (2022). Maximum sampled conditional likelihood for informative subsampling. \emph{Journal of machine learning research}, \strong{23}(332), 1-50.
}
