% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spEigen.R
\name{spEigen}
\alias{spEigen}
\title{Sparse Spectral Decomposition of a Matrix}
\usage{
spEigen(X, q = 1, rho = 0.5, data = FALSE, d = NA, V = NA,
  thres = 1e-09)
}
\arguments{
\item{X}{m-by-m covariance matrix or n-by-m data matrix (n samples, m variables). Both real and complex matrices are accepted.}

\item{q}{number of eigenvectors to be estimated.}

\item{rho}{sparsity weight factor. Any nonnegative number (suggested range [0,1]).}

\item{data}{boolean variable. If \code{TRUE}, \code{X} is treated as a data matrix, else as a covariance matrix (default).}

\item{d}{vector with q weights. The default value is \code{seq(from = 1, to = 0.5, length.out = q)}.}

\item{V}{initial m-by-q matrix point. If not provided, the eigenvectors of the sample covariance matrix are used.}

\item{thres}{threshold value. All the entries of the sparse eigenvectors less or equal to \code{thres} are set to 0. The default value is \code{1e-9}.}
}
\value{
A list with the following components:
\item{\code{vectors}}{m-by-q matrix, columns corresponding to the q leading sparse eigenvectors.}
\item{\code{standard_vectors}}{m-by-q matrix, columns corresponding to standard (non-sparse) leading eigenvectors.}
\item{\code{values}}{vector with the q leading eigenvalues (in decreasing order).}
}
\description{
Computes sparse (orthogonal) eigenvectors of covariance matrix or directly of data matrix.
}
\examples{
library(sparseEigen)
n <- 100  # samples
m <- 500  # dimension
q <- 3  # number of sparse eigenvectors to be estimated
sp_card <- 0.1*m  # sparsity of each eigenvector

# generate covariance matrix with sparse eigenvectors
V <- matrix(0, m, q)
V[cbind(seq(1, q*sp_card), rep(1:q, each = sp_card))] <- 1/sqrt(sp_card)
V <- cbind(V, matrix(rnorm(m*(m-q)), m, m-q))
V <- qr.Q(qr(V))  # orthogonalize eigenvectors
lmd <- c(100*seq(from = q, to = 1), rep(1, m-q))  # generate eigenvalues
R <- V \%*\% diag(lmd) \%*\% t(V)  # covariance matrix

# generate data
X <- MASS::mvrnorm(n, rep(0, m), R)  # random data with underlying sparse structure

# standardand sparse eigenvectors
res_standard <- eigen(cov(X))
res_sparse <- spEigen(cov(X), q)

# show inner product between estimated eigenvectors and originals (the closer to 1 the better)
abs(diag(t(res_standard$vectors) \%*\% V[, 1:q]))  #for standard estimated eigenvectors
abs(diag(t(res_sparse$vectors) \%*\% V[, 1:q]))    #for sparse estimated eigenvectors
}
\references{
K. Benidis, Y. Sun, P. Babu, D. P. Palomar, "Orthogonal Sparse PCA and Covariance Estimation via Procrustes Reformulation,"
\emph{IEEE Transactions on Signal Processing}, vol. 64, no. 23, pp. 6211-6226, Dec. 2016.
}
\author{
Konstantinos Benidis and Daniel P. Palomar
}
