% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{colorize_las}
\alias{colorize_las}
\title{Colorize a LAS object using multiple methods}
\usage{
colorize_las(
  las,
  method = "attr",
  attribute_name = NULL,
  palette = c("black", "white"),
  raster_path = NULL,
  radius = 1,
  num_directions = 60,
  kernel_size = 5,
  pixel_size = 0.1,
  num_samples = 16,
  ncpu = 4
)
}
\arguments{
\item{las}{LAS object to colorize}

\item{method}{Character string specifying the coloring method: "attr" for attribute-based,
"rgb" for raster-based, or "pcv" for ambient occlusion. Default is "attr".}

\item{attribute_name}{Character string specifying the attribute name (required for method="attr").
The attribute must exist in the LAS data.}

\item{palette}{Character vector of at least two colors for the color ramp (used with method="attr", "pcv", or "ssao").
Colors can be hex codes (e.g., "#FF0000") or named colors (e.g., "red"). Default is grayscale.}

\item{raster_path}{Character string or vector of paths to raster files (required for method="rgb").
Can be a single RGB raster or three separate rasters for R, G, and B channels.}

\item{radius}{Numeric radius for neighborhood search in PCV calculation (method="pcv"). Default is 1.0.}

\item{num_directions}{Integer number of directional rays for PCV calculation (method="pcv"). Default is 60.}

\item{kernel_size}{Integer kernel size in pixels for SSAO sampling (method="ssao"). Default is 5.}

\item{pixel_size}{Numeric resolution of the depth map in spatial units (method="ssao"). Default is 0.1.}

\item{num_samples}{Integer number of samples per point for SSAO (method="ssao"). Default is 16.}

\item{ncpu}{Integer number of CPUs to use for parallel processing (method="pcv" or "ssao"). Default is 4.}
}
\value{
A LAS object with updated R, G, and B fields based on the selected method.
}
\description{
Colors a LAS object using one of three methods: attribute-based coloring,
raster-based RGB coloring, or PCV (Portion de Ciel Visible) ambient occlusion.
}
\details{
The function supports four coloring methods:
\describe{
\item{attr}{Attribute-based coloring: normalizes attribute values and maps them to colors using the palette.}
\item{rgb}{Raster-based coloring: extracts RGB values from georeferenced raster(s) that align with the point cloud.
Requires matching CRS between LAS and raster. Can use a single 3-band RGB raster or three separate rasters.}
\item{pcv}{PCV (Portion de Ciel Visible): computes 3D ambient occlusion by calculating sky visibility for each point.
Based on the algorithm from Duguet & Girardeau-Montaut (2004). More accurate but slower than SSAO.}
\item{ssao}{SSAO (Screen Space Ambient Occlusion): fast ambient occlusion using 2D depth map techniques.
Projects points to a depth buffer and calculates occlusion based on depth differences. Much faster than PCV.}
}
}
\examples{
\donttest{

LASfile <- system.file("extdata", "ALS_Clip.laz", package="spanner")
las <- readLAS(LASfile, select = "xyz")

# Attribute-based coloring
las_colored <- colorize_las(las, method="attr", attribute_name="Z",
                            palette=c("blue", "green", "yellow", "red"))

# Raster-based coloring with RGB file
rgb_file <- system.file("extdata", "UAS_Clip_RGB.tif", package="spanner")
las_colored <- colorize_las(las, method="rgb", raster_path=rgb_file)

# PCV ambient occlusion (slow, high quality)
las_colored <- colorize_las(las, method="pcv", radius=1.0,
                            num_directions=30, palette=c("black", "white"))

# SSAO ambient occlusion (faster alternative to PCV)
las_colored <- colorize_las(las, method="ssao", pixel_size=0.1,
                            kernel_size=5, num_samples=16, palette=c("black", "white"), ncpu=8)
}

}
