#' Domains Resource
#'
#' @description
#' R6 class providing access to domain operations.
#'
#' @keywords internal
DomainsResource <- R6::R6Class(
  "DomainsResource",
  public = list(
    #' @description
    #' Create a new DomainsResource.
    #' @param base_req Base httr2 request object.
    initialize = function(base_req) {
      private$.base_req <- base_req
    },

    #' @description
    #' List all domains.
    #'
    #' @param include_stats Include concept counts and vocabulary coverage. Default `FALSE`.
    #'
    #' @returns Domain list.
    list = function(include_stats = FALSE) {
      params <- list()

      if (isTRUE(include_stats)) {
        params$include_stats <- "true"
      }

      perform_get(private$.base_req, "domains", query = params)
    },

    #' @description
    #' Get concepts in a domain.
    #'
    #' @param domain_id The domain ID.
    #' @param vocabulary_ids Filter by vocabularies.
    #' @param standard_only Only standard concepts. Default `FALSE`.
    #' @param include_invalid Include invalid/deprecated concepts. Default `FALSE`.
    #' @param page Page number. Default 1.
    #' @param page_size Results per page. Default 50.
    #'
    #' @returns Paginated concepts.
    concepts = function(domain_id,
                        vocabulary_ids = NULL,
                        standard_only = FALSE,
                        include_invalid = FALSE,
                        page = 1,
                        page_size = 50) {
      checkmate::assert_string(domain_id, min.chars = 1)
      pag <- validate_pagination(page, page_size)

      params <- list(
        page = pag$page,
        page_size = pag$page_size
      )

      if (!is.null(vocabulary_ids)) {
        params$vocabulary_ids <- join_params(vocabulary_ids)
      }
      if (isTRUE(standard_only)) {
        params$standard_only <- "true"
      }
      if (isTRUE(include_invalid)) {
        params$include_invalid <- "true"
      }

      perform_get(
        private$.base_req,
        paste0("domains/", domain_id, "/concepts"),
        query = params
      )
    },

    #' @description
    #' Print resource information.
    print = function() {
      cat("<OMOPHub DomainsResource>\n")
      cat("  Methods: list, concepts\n")
      invisible(self)
    }
  ),
  private = list(
    .base_req = NULL
  )
)
