#' myCRAN
#'
#' @import cranlogs
#' @import grDevices
#' @import graphics
#' @import pkgsearch
#' @import plotrix
#' @import lubridate
#' @importFrom("stats", "lm")
#'
#' @description wrapper for cran_downloads() to plot counts and cumulative counts
#'	of downloads of CRAN packages
#'
#' @param packages character vector of names of packages
#' @param author character string of name of target author
#' @param from date parameter for cran_downloads()
#' @param when date parameter for cran_downloads()
#' @param plotNew Boolean if TRUE start a new plot window
#' @param plotWidth integer width parameter for dev.new()
#' @param regress Boolean if TRUE perform linear regression for each package
#' @param printout Boolean if TRUE print out summary data for the last several days
#'
#' @details Use either packages or author as the input parameter.
#' When author is given, the function advanced_search() queries CRAN
#' to return a list of the names of all packages by that author.
#'
#' @examples
#' \dontrun{
#' 	packages<-c("timeLineGraphics","textBoxPlacement","SherlockHolmes","myCRAN")
#' 	author = "Zeeberg"
#' 	from<-"2023-01-01"
#' 	myCRAN(author=author,from=from,plotNew=TRUE)
#' 	myCRAN(packages=packages,when="last-week",plotNew=FALSE)
#' 	}
#'
#' @return returns no values, but has side effect of generating plots
#'   of daily and cumulative number of downloads of your packages
#'
#' @export
myCRAN<-
  myCRAN<-
  function (packages = NULL, author = NULL, from, when, plotNew = TRUE, 
            plotWidth = 12,regress=TRUE,printout=TRUE) 
  {
    l <- list()
    s <- vector("integer", length(packages))
    if (!is.null(author)) 
      packages <- pkgsearch::advanced_search(Author = author, 
                                             size = 100)$package
    for (pack in packages) {
      l[[pack]] <- cran_downloads(pack, from = from, when = when)
    }
    max <- 0
    for (pack in names(l)) {
      max <- max(max, l[[pack]][, "count"])
      s[pack] <- sum(l[[pack]][, "count"])
    }
    ylim <- c(0, max)
    colors <- c("red", "green", "blue", "orange", "magenta", 
                "pink", "#469990", "brown", "beige", "maroon", "#aaffc3", 
                "#808000", "#ffd8b1", "navy", "grey", "black", "lavender", 
                "yellow", "cyan", "#bfef45", "purple")
    xlabel <- "date"
    ylabel <- "counts"
    title <- "My R Packages Downloads"
    if (plotNew) 
      dev.new(width = plotWidth, unit = "in")
    plot(x = l[[packages[1]]][, "date"], y = l[[packages[1]]][, 
                                                              "count"], ylim = ylim, type = "l", xlab = xlabel, ylab = ylabel, 
         main = title)
    i <- 1
    for (pack in names(l)) {
      lines(x = l[[packages[1]]][, "date"], y = l[[pack]][, 
                                                          "count"], col = colors[i])
      i <- i + 1
    }
    leg <- sprintf("%s %d", names(l), s[names(l)])
    legend("topleft", legend = leg, fill = colors)
    lc <- list()
    max <- 0
    for (pack in names(l)) max <- max(max, sum(l[[pack]][, "count"]))
    ylim <- c(0, max)
    total <- vector("numeric", nrow(l[[names(l)[1]]]))
    for (pack in names(l)) {
      lc[[pack]]$cum <- vector("integer", nrow(l[[pack]]))
      lc[[pack]]$cum[1] <- l[[pack]][1, "count"]
      for (i in 2:nrow(l[[pack]])) {
        lc[[pack]]$cum[i] <- lc[[pack]]$cum[i - 1] + l[[pack]][i, 
                                                               "count"]
        total[i] <- total[i] + lc[[pack]]$cum[i]
      }
    }
    
    if(regress)
      for (pack in names(l)) {    	
        start<-min(which(lc[[pack]]$cum>0))
        print(pack)
        yregress<-lc[[pack]]$cum[(start-1):length(lc[[pack]]$cum)]
        ly<-length(yregress)
        xregress<-1:ly
        model<-lm(yregress ~ xregress)
        su<-summary(model)
        print(su)
        plot(yregress ~ xregress,main=pack,xlab="date",ylab="downloads")
        abline(model,col="red")
        print(names(su))
        print(colnames(su$coefficients))
        print(su$coefficients[,"Estimate"])
        print(su$coefficients["xregress","Estimate"])
        textboxx<-ly/2.5
        textboxy<-max(yregress)/1.25
        m<-matrix(nrow=2,ncol=2)
        colnames(m)<-c("Estimate","Std. Error")
        rownames(m)<-c("Intercept","Slope")
        m[1,1]<-round(su$coefficients["(Intercept)","Estimate"],2)
        m[1,2]<-round(su$coefficients["(Intercept)","Std. Error"],2)
        m[2,1]<-round(su$coefficients["xregress","Estimate"],2)
        m[2,2]<-round(su$coefficients["xregress","Std. Error"],2)
        
        addtable2plot(x=textboxx,y=textboxy,table=m,display.colnames=TRUE,display.rownames=TRUE,bty="o",hlines=TRUE,vlines=TRUE,bg="yellow",text.col="red")
        
        m<-matrix(nrow=1,ncol=2)
        colnames(m)<-c("r.squared","adj.r.squared")
        rownames(m)<-"model"
        m[1,1]<-round(su$r.squared,2)
        m[1,2]<-round(su$adj.r.squared,2)
        addtable2plot(x=textboxx,y=max(yregress)/1.50,table=m,display.colnames=TRUE,display.rownames=TRUE,bty="o",hlines=TRUE,vlines=TRUE,bg="yellow",text.col="red")
        
        rl<-readline(prompt="enter <return> for next regression or x to break from regression:  ")
        if(rl=="x")
          break
      }
    
    par(mar = c(5, 4, 4, 4) + 0.3)
    ylabel <- "cumulative counts"
    plot(x = l[[packages[1]]][, "date"], lc[[names(l)[1]]]$cum, 
         ylim = ylim, type = "l", xlab = xlabel, ylab = ylabel, 
         main = title)
    i <- 1
    for (pack in names(l)) {    	
      lines(x = l[[packages[1]]][, "date"], y = lc[[pack]]$cum, 
            col = colors[i])
      i <- i + 1
    }
    par(new = TRUE)
    ylim <- c(0, max(total))
    if(printout) {
      n<-30
      cat("following tabulation shows that CRAN updating of download counts is periodic and irregular and delayed by a day or two\n\n")
      cat("curr total   prev total  diff    curr date\n")
      for(ii in 1:n) {
        dt<-l[[packages[1]]][, "date"][(length(total)-n+ii)]
        tt0<-total[(length(total)-n+ii)]
        tt1<-total[(length(total)-n+ii-1)]
        #print(c(ii,n,dt,tt0,tt1,tt0-tt1),quote=FALSE)
        cat(" ",tt0,"       ",tt1," ",tt0-tt1,"\t\t")
        message(as.Date(dt,origin="1970-1-1"))
      }
    }
    
    plot(x = l[[packages[1]]][, "date"], y = total, col = "black", 
         type = "l", lwd = 5, axes = FALSE, ylim = ylim, bty = "n", 
         xlab = "", ylab = "")
    mtext("total", side = 4)
    axis(4, ylim = ylim, las = 1)
    legend("topleft", legend = leg, fill = colors)
  }
