#' Plot mean methylation profiles for a selected DMR
#'
#' @description
#' Visualize a single DMR detected by \code{mmcmcBayes()} by plotting the mean
#' M-values across CpG sites in the region for the cancer and normal groups.
#'
#' @details
#' This function takes the DMR table returned by \code{mmcmcBayes()} together 
#' with the original \code{cancer_data} and \code{normal_data} matrices used 
#' in the analysis. It selects one DMR by index and computes, for each CpG site 
#' in the region, the mean M-value across cancer samples and across normal samples. 
#' These two mean profiles are then plotted against the CpG index within the region.
#'
#' The input data frames \code{cancer_data} and \code{normal_data} are expected
#' to have the same CpG sites in the same order, with at least the columns
#' \code{CpG_ID} and \code{Chromosome}, followed by one column per sample
#' containing M-values. Sample columns are identified automatically as those
#' whose names start with \code{"M_sample"}.
#' 
#' CpG sites whose mean M-value is \code{NA} or non finite in either group are
#' removed prior to plotting. The x axis index therefore refers to the filtered
#' CpG positions within the selected region.
#'
#' @param dmr_table A data frame of DMRs, typically the result of
#'   \code{mmcmcBayes()} with \code{return_mcmc = FALSE}.
#' @param cancer_data A data frame of cancer group methylation data in the
#'   format used by \code{mmcmcBayes()} (rows = CpGs, columns = metadata +
#'   M-values for cancer samples).
#' @param normal_data A data frame of normal-group methylation data in the same
#'   format and CpG ordering as \code{cancer_data}.
#' @param dmr_index Integer index of the DMR to plot (row index in
#'   \code{dmr_table}). Defaults to 1.
#' @param main Optional main title for the plot. If \code{NULL}, a title is
#'   constructed from the region information.
#' @param ... Additional arguments passed to \code{plot()}.
#'
#' @return
#' The function is called for its side effect of creating a plot. It returns
#' (invisibly) a list containing the CpG IDs in the region and the mean
#' profiles for cancer and normal groups.
#'
#' @examples
#' \donttest{
#' data(cancer_demo)
#' data(normal_demo)
#'
#' set.seed(2021)
#' mcmc <- list(nburn = 1000, niter = 2000, thin = 1)
#'
#' dmr_res <- mmcmcBayes(
#'   cancer_demo, normal_demo,
#'   stage         = 1,
#'   max_stages    = 2,
#'   num_splits    = 5,
#'   mcmc          = mcmc,
#'   priors_cancer = NULL,
#'   priors_normal = NULL,
#'   bf_thresholds = c(0.5, 0.8, 1.05)
#' )
#'
#' plot_dmr_region(dmr_res, cancer_demo, normal_demo, dmr_index = 1)
#' }
#'
#' @importFrom graphics plot lines legend abline
#' @export
plot_dmr_region <- function(dmr_table,
                            cancer_data,
                            normal_data,
                            dmr_index = 1,
                            main = NULL,
                            ...) {
  
  if (is.null(dmr_table) || nrow(dmr_table) == 0) {
    stop("`dmr_table` is NULL or has zero rows; no DMRs to plot.", call. = FALSE)
  }
  
  ## --- Extract DMR ---
  dmr <- dmr_table[dmr_index, ]
  start_id <- as.character(dmr$Start_CpG)
  end_id   <- as.character(dmr$End_CpG)
  
  start_idx <- which(cancer_data$CpG_ID == start_id)
  end_idx   <- which(cancer_data$CpG_ID == end_id)
  
  if (length(start_idx) != 1 || length(end_idx) != 1)
    stop("Start/End CpG not uniquely found.")
  
  if (end_idx < start_idx) {
    tmp <- start_idx; start_idx <- end_idx; end_idx <- tmp
  }
  
  region_idx <- seq(start_idx, end_idx)
  
  ## --- Extract sample matrices ---
  cancer_cols <- grep("^M_sample", names(cancer_data), value = TRUE)
  normal_cols <- grep("^M_sample", names(normal_data), value = TRUE)
  
  cancer_mat <- as.matrix(cancer_data[region_idx, cancer_cols, drop = FALSE])
  normal_mat <- as.matrix(normal_data[region_idx, normal_cols, drop = FALSE])
  
  mean_cancer <- rowMeans(cancer_mat, na.rm = TRUE)
  mean_normal <- rowMeans(normal_mat, na.rm = TRUE)
  
  ## Remove CpGs with NA in either group
  keep <- which(is.finite(mean_cancer) & is.finite(mean_normal))
  if (length(keep) == 0)
    stop("All CpGs filtered out due to NA.")
  
  mean_cancer <- mean_cancer[keep]
  mean_normal <- mean_normal[keep]
  
  x <- seq_along(keep)
  ylim <- range(c(mean_cancer, mean_normal))
  
  ## Title
  if (is.null(main)) {
    main <- sprintf(
      "DMR %d: chr%s [%s-%s], Decision_Value = %.3f",
      dmr_index, dmr$Chromosome,
      start_id, end_id, dmr$Decision_Value
    )
  }
  
  

  
  ## --- Main plot ---
  plot(x, mean_cancer, type = "o", pch = 16, lwd = 2,
       ylim = ylim,
       xlab = "CpG index in region",
       ylab = "Mean M-value",
       main = main,
       bty  = "l",
       ...)
  
  abline(h = 0, col = "grey85")
  lines(x, mean_normal, type = "o", pch = 1, lwd = 2, lty = 2)
  
  ## --- Legend BELOW x-axis ---
  legend("bottom",
         xpd = TRUE,       # allow drawing outside plot region
         horiz = TRUE,
         legend = c("Cancer (mean)", "Normal (mean)"),
         lty    = c(1, 2),
         pch    = c(16, 1),
         bty    = "n",
         ...)
  
  invisible(list(
    CpG_IDs      = cancer_data$CpG_ID[region_idx][keep],
    mean_cancer  = mean_cancer,
    mean_normal  = mean_normal
  ))
}
