% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plsda.R
\name{plsda}
\alias{plsda}
\title{Partial Least Squares Discriminant Analysis}
\usage{
plsda(
  x,
  c,
  ncomp = min(nrow(x) - 1, ncol(x), 20),
  center = TRUE,
  scale = FALSE,
  cv = NULL,
  exclcols = NULL,
  exclrows = NULL,
  x.test = NULL,
  c.test = NULL,
  method = "simpls",
  lim.type = "ddmoments",
  alpha = 0.05,
  gamma = 0.01,
  info = "",
  ncomp.selcrit = "min",
  classname = NULL,
  cv.scope = "local"
)
}
\arguments{
\item{x}{matrix with predictors.}

\item{c}{vector with class membership (should be either a factor with class names/numbers in case of
multiple classes or a vector with logical values in case of one class model).}

\item{ncomp}{maximum number of components to calculate.}

\item{center}{logical, center or not predictors and response values.}

\item{scale}{logical, scale (standardize) or not predictors and response values.}

\item{cv}{cross-validation settings (see details).}

\item{exclcols}{columns of x to be excluded from calculations (numbers, names or vector with logical values)}

\item{exclrows}{rows to be excluded from calculations (numbers, names or vector with logical values)}

\item{x.test}{matrix with predictors for test set.}

\item{c.test}{vector with reference class values for test set (same format as calibration values).}

\item{method}{method for calculating PLS model.}

\item{lim.type}{which method to use for calculation of critical limits for residual distances (see details)}

\item{alpha}{significance level for extreme limits for T2 and Q disances.}

\item{gamma}{significance level for outlier limits for T2 and Q distances.}

\item{info}{short text with information about the model.}

\item{ncomp.selcrit}{criterion for selecting optimal number of components (\code{'min'} for first local minimum of
RMSECV and \code{'wold'} for Wold's rule.)}

\item{classname}{name (label) of class in case if PLS-DA is used for one-class discrimination model. In this case
it is expected that parameter `c` will be a vector with logical values.}

\item{cv.scope}{scope for center/scale operations inside CV loop: 'global' — using globally computed mean and std
or 'local' — recompute new for each local calibration set.}
}
\value{
Returns an object of \code{plsda} class with following fields (most inherited from class
\code{pls}):
\item{ncomp }{number of components included to the model.}
\item{ncomp.selected }{selected (optimal) number of components.}
\item{xloadings }{matrix with loading values for x decomposition.}
\item{yloadings }{matrix with loading values for y (c)  decomposition.}
\item{weights }{matrix with PLS weights.}
\item{coeffs }{matrix with regression coefficients calculated for each component.}
\item{info }{information about the model, provided by user when build the model.}
\item{calres }{an object of class \code{\link{plsdares}} with PLS-DA results for a calibration
data.}
\item{testres }{an object of class \code{\link{plsdares}} with PLS-DA results for a test data,
if it was provided.}
\item{cvres }{an object of class \code{\link{plsdares}} with PLS-DA results for cross-validation,
if this option was chosen.}
}
\description{
\code{plsda} is used to calibrate, validate and use of partial least squares discrimination
analysis (PLS-DA) model.
}
\details{
The \code{plsda} class is based on \code{pls} with extra functions and plots covering
classification functionality. All plots for \code{pls} can be used. E.g. of you want to see the
real predicted values (y in PLS) instead of classes use \code{plotPredictions.pls(model)} instead
of \code{plotPredictions(model)}.

Cross-validation settings, \code{cv}, can be a number or a list. If \code{cv} is a number, it
will be used as a number of segments for random cross-validation (if \code{cv = 1}, full
cross-validation will be preformed). If it is a list, the following syntax can be used:
\code{cv = list('rand', nseg, nrep)} for random repeated cross-validation with \code{nseg}
segments and \code{nrep} repetitions or \code{cv = list('ven', nseg)} for systematic splits
to \code{nseg} segments ('venetian blinds').

Calculation of confidence intervals and p-values for regression coefficients are available
only by jack-knifing so far. See help for \code{\link{regcoeffs}} objects for details.
}
\examples{
### Examples for PLS-DA model class

library(mdatools)

## 1. Make a PLS-DA model with full cross-validation and show model overview

# make a calibration set from iris data (3 classes)
# use names of classes as class vector
x.cal = iris[seq(1, nrow(iris), 2), 1:4]
c.cal = iris[seq(1, nrow(iris), 2), 5]

model = plsda(x.cal, c.cal, ncomp = 3, cv = 1, info = 'IRIS data example')
model = selectCompNum(model, 1)

# show summary and basic model plots
# misclassification will be shown only for first class
summary(model)
plot(model)

# summary and model plots for second class
summary(model, nc = 2)
plot(model, nc = 2)

# summary and model plot for specific class and number of components
summary(model, nc = 3, ncomp = 3)
plot(model, nc = 3, ncomp = 3)

## 2. Show performance plots for a model
par(mfrow = c(2, 2))
plotSpecificity(model)
plotSensitivity(model)
plotMisclassified(model)
plotMisclassified(model, nc = 2)
par(mfrow = c(1, 1))

## 3. Show both class and y values predictions
par(mfrow = c(2, 2))
plotPredictions(model)
plotPredictions(model, res = "cal", ncomp = 2, nc = 2)
plotPredictions(structure(model, class = "regmodel"))
plotPredictions(structure(model, class = "regmodel"), ncomp = 2, ny = 2)
par(mfrow = c(1, 1))

## 4. All plots from ordinary PLS can be used, e.g.:
par(mfrow = c(2, 2))
plotXYScores(model)
plotYVariance(model)
plotXResiduals(model)
plotRegcoeffs(model, ny = 2)
par(mfrow = c(1, 1))

}
\seealso{
Specific methods for \code{plsda} class:
\tabular{ll}{
 \code{print.plsda} \tab prints information about a \code{pls} object.\cr
 \code{summary.plsda} \tab shows performance statistics for the model.\cr
 \code{plot.plsda} \tab shows plot overview of the model.\cr
 \code{\link{predict.plsda}} \tab applies PLS-DA model to a new data.\cr
}

Methods, inherited from \code{classmodel} class:
\tabular{ll}{
 \code{\link{plotPredictions.classmodel}} \tab shows plot with predicted values.\cr
 \code{\link{plotSensitivity.classmodel}} \tab shows sensitivity plot.\cr
 \code{\link{plotSpecificity.classmodel}} \tab shows specificity plot.\cr
 \code{\link{plotMisclassified.classmodel}} \tab shows misclassified ratio plot.\cr
}

See also methods for class \code{\link{pls}}.
}
\author{
Sergey Kucheryavskiy (svkucheryavski@gmail.com)
}
