% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regularizeSmoothSEMInterface.R
\name{smoothAdaptiveLasso}
\alias{smoothAdaptiveLasso}
\title{smoothAdaptiveLasso}
\usage{
smoothAdaptiveLasso(
  lavaanModel,
  regularized,
  weights = NULL,
  lambdas,
  epsilon,
  tau,
  modifyModel = lessSEM::modifyModel(),
  control = lessSEM::controlBFGS()
)
}
\arguments{
\item{lavaanModel}{model of class lavaan}

\item{regularized}{vector with names of parameters which are to be regularized.
If you are unsure what these parameters are called, use
getLavaanParameters(model) with your lavaan model object}

\item{weights}{labeled vector with weights for each of the parameters in the
model. If you are unsure what these parameters are called, use
getLavaanParameters(model) with your lavaan model object. If set to NULL,
the default weights will be used: the inverse of the absolute values of
the unregularized parameter estimates}

\item{lambdas}{numeric vector: values for the tuning parameter lambda}

\item{epsilon}{epsilon > 0; controls the smoothness of the approximation. Larger values = smoother}

\item{tau}{parameters below threshold tau will be seen as zeroed}

\item{modifyModel}{used to modify the lavaanModel. See ?modifyModel.}

\item{control}{used to control the optimizer. This element is generated with
the controlBFGS function. See ?controlBFGS for more details.}
}
\value{
Model of class regularizedSEM
}
\description{
This function allows for regularization of models built in lavaan with the
smooth adaptive lasso penalty. The returned object is an S4 class; its elements can be accessed
with the "@" operator (see examples).
}
\details{
For more details, see:
\enumerate{
\item Zou, H. (2006). The Adaptive Lasso and Its Oracle Properties.
Journal of the American Statistical Association, 101(476), 1418–1429.
https://doi.org/10.1198/016214506000000735
\item Jacobucci, R., Grimm, K. J., & McArdle, J. J. (2016).
Regularized Structural Equation Modeling. Structural Equation Modeling:
A Multidisciplinary Journal, 23(4), 555–566. https://doi.org/10.1080/10705511.2016.1154793
\item Lee, S.-I., Lee, H., Abbeel, P., & Ng, A. Y. (2006).
Efficient L1 Regularized Logistic Regression. Proceedings of the
Twenty-First National Conference on Artificial Intelligence (AAAI-06), 401–408.
}
}
\examples{
library(lessSEM)

# Identical to regsem, lessSEM builds on the lavaan
# package for model specification. The first step
# therefore is to implement the model in lavaan.

dataset <- simulateExampleData()

lavaanSyntax <- "
f =~ l1*y1 + l2*y2 + l3*y3 + l4*y4 + l5*y5 +
     l6*y6 + l7*y7 + l8*y8 + l9*y9 + l10*y10 +
     l11*y11 + l12*y12 + l13*y13 + l14*y14 + l15*y15
f ~~ 1*f
"

lavaanModel <- lavaan::sem(lavaanSyntax,
                           data = dataset,
                           meanstructure = TRUE,
                           std.lv = TRUE)

# Regularization:

# names of the regularized parameters:
regularized = paste0("l", 6:15)

# define adaptive lasso weights:
# We use the inverse of the absolute unregularized parameters
# (this is the default in adaptiveLasso and can also specified
# by setting weights = NULL)
weights <- 1/abs(getLavaanParameters(lavaanModel))
weights[!names(weights) \%in\% regularized] <- 0

lsem <- smoothAdaptiveLasso(
  # pass the fitted lavaan model
  lavaanModel = lavaanModel,
  regularized = regularized,
  weights = weights,
  epsilon = 1e-10,
  tau = 1e-4,
  lambdas = seq(0,1,length.out = 50))

# use the plot-function to plot the regularized parameters:
plot(lsem)

# the coefficients can be accessed with:
coef(lsem)

# elements of lsem can be accessed with the @ operator:
lsem@parameters[1,]

# AIC and BIC:
AIC(lsem)
BIC(lsem)

# The best parameters can also be extracted with:
coef(lsem, criterion = "AIC")
coef(lsem, criterion = "BIC")
}
