% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/augment.R
\name{augment}
\alias{augment}
\title{Augment SEM data with predictions, residuals, SEs/CIs, and ordinal extras}
\usage{
augment(
  fit,
  data = NULL,
  info = NULL,
  yhat = TRUE,
  resid = TRUE,
  ci = TRUE,
  level = 0.95,
  se_yhat = TRUE,
  ystar = TRUE,
  pr = TRUE,
  se_fs = TRUE,
  vcov_type = NULL,
  col_layout = c("by_type", "by_item"),
  prefix_ystar = ".ystar_",
  prefix_yhat = ".yhat_",
  prefix_pr = ".pr_",
  prefix_ci = c(".yhat_lwr_", ".yhat_upr_"),
  prefix_resid = ".resid_",
  prefix_se_fs = ".se_",
  prefix_se_yhat = ".se_yhat_",
  sep = "__"
)
}
\arguments{
\item{fit}{A fitted \code{lavaan} model object.}

\item{data}{Optional factor-score output to reuse. Either a \code{data.frame}
(single-group) or a \code{list} of \code{data.frame}s (per group) as returned
by \code{\link{lavPredict_parallel}()}. If \code{NULL}, it is computed once
and reused internally.}

\item{info}{Optional \code{\link{model_info}()} list; if \code{NULL}, it is computed.}

\item{yhat}{Logical; include predicted observed values. Default \code{TRUE}.}

\item{resid}{Logical; include residuals (\code{obs - yhat}). Default \code{TRUE}.}

\item{ci}{Logical; include delta-method confidence intervals for \code{yhat}.
Default \code{TRUE}.}

\item{level}{Confidence level for \code{ci}. Default \code{0.95}.}

\item{se_yhat}{Logical; include delta-method standard errors of \code{yhat}.
Default \code{TRUE}.}

\item{ystar}{Logical; for ordinal items, include latent linear predictors \eqn{y^*}.
Default \code{TRUE}.}

\item{pr}{Logical; for ordinal items, include per-category probabilities.
Default \code{TRUE}.}

\item{se_fs}{Logical; request factor-score SEs from \code{lavPredict_parallel()}
for continuous-only models. Ignored when the model contains any ordinal
indicators. Default \code{TRUE}.}

\item{vcov_type}{Optional \code{\link[stats]{vcov}} type passed to the continuous
branch when computing robust SEs/CIs for predictions (e.g., \code{"sandwich"}).}

\item{col_layout}{Column layout for the augmented output; either
\code{"by_type"} (observed, \code{y*}, \code{yhat}, CI lower/upper, \code{se_yhat},
residuals, probabilities) or \code{"by_item"} (per-item blocks). Default
\code{c("by_type","by_item")} with \code{match.arg()} semantics.}

\item{prefix_ystar}{Character prefix for latent linear predictor columns
(ordinal), e.g., \code{".ystar_"}. Default \code{".ystar_"}.}

\item{prefix_yhat}{Character prefix for predicted observed values, e.g.,
\code{".yhat_"}. Default \code{".yhat_"}.}

\item{prefix_pr}{Character prefix for ordinal category-probability columns,
e.g., \code{".pr_"}. Default \code{".pr_"}.}

\item{prefix_ci}{Length-2 character vector with lower/upper prefixes for
prediction intervals, e.g., \code{c(".yhat_lwr_", ".yhat_upr_")}. Default \code{c(".yhat_lwr_", ".yhat_upr_")}.}

\item{prefix_resid}{Character prefix for residual columns, e.g., \code{".resid_"}.
Default \code{".resid_"}.}

\item{prefix_se_fs}{Character prefix for factor-score SE columns (continuous-only),
e.g., \code{".se_"}. Default \code{".se_"}.}

\item{prefix_se_yhat}{Character prefix for prediction SE columns, e.g.,
\code{".se_yhat_"}. Default \code{".se_yhat_"}.}

\item{sep}{Separator used in probability column names between category and item,
e.g., \code{"__"} giving names like \code{".pr_2__Item1"}. Default \code{"__"}.}
}
\value{
A tibble-like \code{data.frame}. Columns appear in the following order and with the following rules:
\itemize{
\item \strong{(i) Anchors first}: \code{.rid}, \code{.gid}, \code{.group} always lead the table.
\item \strong{(ii) Original lavPredict columns next}: observed indicators, factor scores, and (optionally) factor-score SEs returned by \code{\link{lavPredict_parallel}()} follow immediately after anchors.
\item \strong{(iii) FS SEs presence}: factor-score SE columns are present \emph{only} when \code{se_fs = TRUE} \emph{and} the model is continuous-only (i.e., contains no ordinal indicators).
\item \strong{(iv) Augmentations}: per measurement type, adds \code{y*} (for ordinal), \code{yhat}, CI lower/upper, \code{se_yhat}, residuals, and (for ordinal) per-category probabilities.
\item \strong{(v) Probability naming}: ordinal probability columns follow the pattern \code{<prefix_pr><cat><sep><item>} (e.g., \code{".pr_3__A1"}).
}
The relative order of the augmentation blocks (observed/\code{y*}/\code{yhat}/CIs/SEs/residuals/probabilities) is controlled by \code{col_layout}.
}
\description{
User-facing wrapper that augments a fitted \pkg{lavaan} model with:
predicted observed values (\emph{yhat}), residuals (\emph{obs - yhat}),
delta-method standard errors and confidence intervals for predictions,
and—when the model includes ordinal indicators—latent linear predictors
(\emph{y*}) and per-category probabilities. Works for continuous-only,
ordinal-only, and mixed models by internally routing to specialized
helpers for each measurement type.
}
\details{
Internally, \code{augment()} delegates work to the internal functions
\code{.augment_continuous()} and \code{.augment_ordinal()}, each optimized for
their respective indicator type. The wrapper automatically detects whether
the fitted model contains continuous, ordinal, or mixed indicators and merges
outputs from both branches as needed.

The function reuses optional inputs:
\itemize{
\item \code{data}: precomputed factor scores (and optional FS SEs) as returned
by \code{\link{lavPredict_parallel}()} to avoid duplicate work.
\item \code{info}: model metadata from \code{\link{model_info}()}.
}
Column naming is controlled by prefix arguments. For ordinal probabilities,
names follow the pattern \code{<prefix_pr><category><sep><item>}, e.g.
\code{".pr_3__A1"} for category 3 of item \code{A1} when \code{sep = "__"}.
The final column order can be arranged either "by type" (all observed, then
all \code{y*}, all \code{yhat}, CIs, SEs, residuals, then probabilities) or
"by item" (grouping each item's block together) via \code{col_layout}.
}
\examples{
# Continuous example
HS.model <- 'visual  =~ x1 + x2 + x3
             textual =~ x4 + x5 + x6
             speed   =~ x7 + x8 + x9'
fit <- lavaan::cfa(HS.model,
                   data = lavaan::HolzingerSwineford1939,
                   meanstructure = TRUE)
augment(fit)

\donttest{
# --- Ordinal example (discretize by quantiles; 5 ordered categories) -------
ord_items <- paste0("x", 1:9)
HS_ord <- lavaan::HolzingerSwineford1939
for (v in ord_items) {
  q <- stats::quantile(HS_ord[[v]], probs = seq(0, 1, length.out = 6), na.rm = TRUE)
  q <- unique(q)  # guard against duplicate cut points
  HS_ord[[v]] <- as.ordered(cut(HS_ord[[v]], breaks = q, include.lowest = TRUE))
}

fit_ord <- lavaan::cfa(
  HS.model,
  data             = HS_ord,
  ordered          = ord_items,
  estimator        = "WLSMV",
  parameterization = "delta",
  meanstructure    = TRUE
)
augment(fit_ord)

# --- Mixed example (x1–x3 ordinal, others continuous) ----------------------
mix_ord <- c("x1","x2","x3")
HS_mix  <- lavaan::HolzingerSwineford1939
for (v in mix_ord) {
  q <- stats::quantile(HS_mix[[v]], probs = seq(0, 1, length.out = 6), na.rm = TRUE)
  q <- unique(q)
  HS_mix[[v]] <- as.ordered(cut(HS_mix[[v]], breaks = q, include.lowest = TRUE))
}

fit_mix <- lavaan::cfa(
  HS.model,
  data             = HS_mix,
  ordered          = mix_ord,
  estimator        = "WLSMV",
  parameterization = "delta",
  meanstructure    = TRUE
)
augment(fit_mix)
}
}
\seealso{
Other lavDiag-augmenters: 
\code{\link{item_data}()},
\code{\link{prepare}()}
}
\concept{lavDiag-augmenters}
