/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5A_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5SMprivate.h" 
#include "H5Iprivate.h"  
#include "H5Fprivate.h"  

typedef struct {
    H5F_t       *f;     
    H5O_ainfo_t *ainfo; 
} H5O_iter_cvt_t;

typedef struct {
    
    const char *name; 

    
    H5A_t *attr; 
} H5O_iter_opn_t;

typedef struct {
    
    H5F_t *f;    
    H5A_t *attr; 

    
    bool found; 
} H5O_iter_wrt_t;

typedef struct {
    
    H5F_t      *f;        
    const char *old_name; 
    const char *new_name; 

    
    bool found; 
} H5O_iter_ren_t;

typedef struct {
    
    H5F_t      *f;    
    const char *name; 

    
    bool found; 
} H5O_iter_rm_t;

typedef struct {
    
    const char *name; 

    
    bool *exists; 
} H5O_iter_xst_t;

static herr_t H5O__attr_to_dense_cb(H5O_t *oh, H5O_mesg_t *mesg, unsigned H5_ATTR_UNUSED sequence,
                                    void *_udata);
static htri_t H5O__attr_find_opened_attr(const H5O_loc_t *loc, H5A_t **attr, const char *name_to_open);
static herr_t H5O__attr_open_cb(H5O_t *oh, H5O_mesg_t *mesg, unsigned sequence, void *_udata);
static herr_t H5O__attr_open_by_idx_cb(const H5A_t *attr, void *_ret_attr);
static herr_t H5O__attr_write_cb(H5O_t *oh, H5O_mesg_t *mesg, unsigned H5_ATTR_UNUSED sequence, void *_udata);
static herr_t H5O__attr_rename_chk_cb(H5O_t H5_ATTR_UNUSED *oh, H5O_mesg_t *mesg,
                                      unsigned H5_ATTR_UNUSED sequence, void *_udata);
static herr_t H5O__attr_rename_mod_cb(H5O_t *oh, H5O_mesg_t *mesg, unsigned H5_ATTR_UNUSED sequence,
                                      void *_udata);
static herr_t H5O__attr_remove_update(const H5O_loc_t *loc, H5O_t *oh, H5O_ainfo_t *ainfo);
static herr_t H5O__attr_remove_cb(H5O_t *oh, H5O_mesg_t *mesg, unsigned H5_ATTR_UNUSED sequence,
                                  void *_udata);
static herr_t H5O__attr_exists_cb(H5O_t H5_ATTR_UNUSED *oh, H5O_mesg_t *mesg,
                                  unsigned H5_ATTR_UNUSED sequence, void *_udata);

static herr_t
H5O__attr_to_dense_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned H5_ATTR_UNUSED sequence,
                      void *_udata )
{
    H5O_iter_cvt_t *udata     = (H5O_iter_cvt_t *)_udata; 
    H5A_t          *attr      = (H5A_t *)mesg->native;    
    herr_t          ret_value = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(mesg);
    assert(udata);
    assert(udata->f);
    assert(udata->ainfo);
    assert(attr);

    
    if (H5A__dense_insert(udata->f, udata->ainfo, attr) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINSERT, H5_ITER_ERROR, "unable to add to dense storage");

    
    
    if (H5O__release_mesg(udata->f, oh, mesg, false) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, H5_ITER_ERROR, "unable to convert into null message");

    
    oh->mesgs_modified = H5O_MODIFY_CONDENSE;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_create(const H5O_loc_t *loc, H5A_t *attr)
{
    H5O_t      *oh = NULL;           
    H5O_ainfo_t ainfo;               
    htri_t      shared_mesg;         
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(attr);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    if (!(H5O_has_chksum(oh) || (H5F_RFIC_FLAGS(loc->file) & H5F_RFIC_UNUSUAL_NUM_UNUSED_NUMERIC_BITS)) &&
        H5T_is_numeric_with_unusual_unused_bits(attr->shared->dt))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL,
                    "creating attribute with unusual datatype, see documentation for "
                    "H5Pset_relax_file_integrity_checks for details.");

    
    if (oh->version > H5O_VERSION_1) {
        bool   new_ainfo = false; 
        htri_t ainfo_exists;      

        
        if ((ainfo_exists = H5A__get_ainfo(loc->file, oh, &ainfo)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
        if (!ainfo_exists) {
            
            ainfo.track_corder    = (bool)((oh->flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED) ? true : false);
            ainfo.index_corder    = (bool)((oh->flags & H5O_HDR_ATTR_CRT_ORDER_INDEXED) ? true : false);
            ainfo.max_crt_idx     = 0;
            ainfo.corder_bt2_addr = HADDR_UNDEF;
            ainfo.nattrs          = 0;
            ainfo.fheap_addr      = HADDR_UNDEF;
            ainfo.name_bt2_addr   = HADDR_UNDEF;

            
            new_ainfo = true;
        } 
        else {
            
            assert(ainfo.nattrs > 0);
            assert(ainfo.track_corder == ((oh->flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED) > 0));
            assert(ainfo.index_corder == ((oh->flags & H5O_HDR_ATTR_CRT_ORDER_INDEXED) > 0));
        } 

        
        if (!H5_addr_defined(ainfo.fheap_addr)) {
            htri_t shareable;    
            size_t raw_size = 0; 

            
            if ((shareable = H5SM_can_share(loc->file, NULL, NULL, H5O_ATTR_ID, attr)) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_BADMESG, FAIL, "can't determine attribute sharing status");
            else if (shareable == false) {
                
                raw_size = (H5O_MSG_ATTR->raw_size)(loc->file, false, attr);
            } 

            
            if (ainfo.nattrs == oh->max_compact || (!shareable && raw_size >= H5O_MESG_MAX_SIZE)) {
                H5O_iter_cvt_t      udata; 
                H5O_mesg_operator_t op;    

                
                if (H5A__dense_create(loc->file, &ainfo) < 0)
                    HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL,
                                "unable to create dense storage for attributes");

                
                udata.f     = loc->file;
                udata.ainfo = &ainfo;

                
                op.op_type  = H5O_MESG_OP_LIB;
                op.u.lib_op = H5O__attr_to_dense_cb;
                if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTCONVERT, FAIL,
                                "error converting attributes to dense storage");
            } 
        }     

        
        ainfo.nattrs++;

        
        if (ainfo.track_corder) {
            
            if (ainfo.max_crt_idx == H5O_MAX_CRT_ORDER_IDX)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTINC, FAIL, "attribute creation index can't be incremented");

            
            attr->shared->crt_idx = ainfo.max_crt_idx++;
        } 
        else
            
            attr->shared->crt_idx = H5O_MAX_CRT_ORDER_IDX;

        
        if (new_ainfo) {
            if (H5O__msg_append_real(loc->file, oh, H5O_MSG_AINFO, H5O_MSG_FLAG_DONTSHARE, 0, &ainfo) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to create new attribute info message");
        } 
        
        else if (H5O__msg_write_real(loc->file, oh, H5O_MSG_AINFO, H5O_MSG_FLAG_DONTSHARE, 0, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update attribute info message");
    } 
    else {
        
        attr->shared->crt_idx = H5O_MAX_CRT_ORDER_IDX;

        
        ainfo.fheap_addr = HADDR_UNDEF;
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (H5A__dense_insert(loc->file, &ainfo, attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to add to dense storage");
    } 
    else
        
        if (H5O__msg_append_real(loc->file, oh, H5O_MSG_ATTR, 0, 0, attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to create new attribute in header");

    
    attr->shared->nrefs += 1;

    
    if ((shared_mesg = H5O_msg_is_shared(H5O_ATTR_ID, attr)) > 0) {
        hsize_t attr_rc; 

        
        if (H5SM_get_refcount(loc->file, H5O_ATTR_ID, &attr->sh_loc, &attr_rc) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't retrieve shared message ref count");

        
        if (attr_rc > 1) {
            if (H5O__attr_delete(loc->file, oh, attr) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute");
        } 
    }     
    else if (shared_mesg < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_WRITEERROR, FAIL, "error determining if message should be shared");

    
    if (H5O_touch_oh(loc->file, oh, false) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_open_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned sequence, void *_udata )
{
    H5O_iter_opn_t *udata     = (H5O_iter_opn_t *)_udata; 
    herr_t          ret_value = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(mesg);
    assert(!udata->attr);

    
    if (strcmp(((H5A_t *)mesg->native)->shared->name, udata->name) == 0) {
        
        if (NULL == (udata->attr = H5A__copy(NULL, (H5A_t *)mesg->native)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, H5_ITER_ERROR, "unable to copy attribute");

        
        if (oh->version == H5O_VERSION_1 || !(oh->flags & H5O_HDR_ATTR_CRT_ORDER_TRACKED))
            udata->attr->shared->crt_idx = sequence;

        
        ret_value = H5_ITER_STOP;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5A_t *
H5O__attr_open_by_name(const H5O_loc_t *loc, const char *name)
{
    H5O_t      *oh = NULL;               
    H5O_ainfo_t ainfo;                   
    H5A_t      *exist_attr      = NULL;  
    H5A_t      *opened_attr     = NULL;  
    htri_t      found_open_attr = false; 
    H5A_t      *ret_value       = NULL;  

    FUNC_ENTER_PACKAGE_TAG(loc->addr)

    
    assert(loc);
    assert(name);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPROTECT, NULL, "unable to load object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "can't check for attribute info message");
    } 

    
    if ((found_open_attr = H5O__attr_find_opened_attr(loc, &exist_attr, name)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "failed in finding opened attribute");
    else if (found_open_attr == true) {
        if (NULL == (opened_attr = H5A__copy(NULL, exist_attr)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, NULL, "can't copy existing attribute");
    } 
    else {
        
        if (H5_addr_defined(ainfo.fheap_addr)) {
            
            if (NULL == (opened_attr = H5A__dense_open(loc->file, &ainfo, name)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, NULL, "can't open attribute");
        } 
        else {
            H5O_iter_opn_t      udata; 
            H5O_mesg_operator_t op;    

            
            udata.name = name;
            udata.attr = NULL;

            
            op.op_type  = H5O_MESG_OP_LIB;
            op.u.lib_op = H5O__attr_open_cb;
            if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, NULL, "error updating attribute");

            
            if (!udata.attr)
                HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, NULL, "can't locate attribute: '%s'", name);

            
            assert(udata.attr);
            opened_attr = udata.attr;
        } 

        
        if (H5T_set_loc(opened_attr->shared->dt, H5F_VOL_OBJ(loc->file), H5T_LOC_DISK) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, NULL, "invalid datatype location");
    } 

    
    ret_value = opened_attr;

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, NULL, "unable to release object header");

    
    if (NULL == ret_value && opened_attr)
        if (H5A__close(opened_attr) < 0)
            HDONE_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, NULL, "can't close attribute");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static herr_t
H5O__attr_open_by_idx_cb(const H5A_t *attr, void *_ret_attr)
{
    H5A_t **ret_attr  = (H5A_t **)_ret_attr; 
    herr_t  ret_value = H5_ITER_STOP;        

    FUNC_ENTER_PACKAGE

    
    assert(attr);
    assert(ret_attr);

    
    if (NULL == (*ret_attr = H5A__copy(NULL, attr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, H5_ITER_ERROR, "can't copy attribute");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5A_t *
H5O__attr_open_by_idx(const H5O_loc_t *loc, H5_index_t idx_type, H5_iter_order_t order, hsize_t n)
{
    H5A_attr_iter_op_t attr_op;                 
    H5A_t             *exist_attr      = NULL;  
    H5A_t             *opened_attr     = NULL;  
    htri_t             found_open_attr = false; 
    H5A_t             *ret_value       = NULL;  

    FUNC_ENTER_PACKAGE

    
    assert(loc);

    
    attr_op.op_type  = H5A_ATTR_OP_LIB;
    attr_op.u.lib_op = H5O__attr_open_by_idx_cb;

    
    if (H5O_attr_iterate_real((hid_t)-1, loc, idx_type, order, n, NULL, &attr_op, &opened_attr) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_BADITER, NULL, "can't locate attribute");

    
    if (opened_attr) {
        if ((found_open_attr = H5O__attr_find_opened_attr(loc, &exist_attr, opened_attr->shared->name)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "failed in finding opened attribute");

        
        if (found_open_attr && exist_attr) {
            if (H5A__close(opened_attr) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, NULL, "can't close attribute");
            if (NULL == (opened_attr = H5A__copy(NULL, exist_attr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, NULL, "can't copy existing attribute");
        }
        else {
            
            if (H5T_set_loc(opened_attr->shared->dt, H5F_VOL_OBJ(loc->file), H5T_LOC_DISK) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, NULL, "invalid datatype location");
        } 
    }     

    
    ret_value = opened_attr;

done:
    
    if (NULL == ret_value && opened_attr)
        if (H5A__close(opened_attr) < 0)
            HDONE_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, NULL, "can't close attribute");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static htri_t
H5O__attr_find_opened_attr(const H5O_loc_t *loc, H5A_t **attr, const char *name_to_open)
{
    hid_t        *attr_id_list = NULL; 
    unsigned long loc_fnum;            
    size_t        num_open_attr;       
    htri_t        ret_value = false;   

    FUNC_ENTER_PACKAGE

    
    if (H5F_get_fileno(loc->file, &loc_fnum) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_BADVALUE, FAIL, "can't get file serial number");

    
    if (H5F_get_obj_count(loc->file, H5F_OBJ_ATTR | H5F_OBJ_LOCAL, false, &num_open_attr) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't count opened attributes");

    
    if (num_open_attr) {
        size_t check_num_attr; 
        size_t u;              

        
        if (NULL == (attr_id_list = (hid_t *)H5MM_malloc(num_open_attr * sizeof(hid_t))))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTALLOC, FAIL, "unable to allocate memory for attribute ID list");

        
        if (H5F_get_obj_ids(loc->file, H5F_OBJ_ATTR | H5F_OBJ_LOCAL, num_open_attr, attr_id_list, false,
                            &check_num_attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get IDs of opened attributes");
        if (check_num_attr != num_open_attr)
            HGOTO_ERROR(H5E_ATTR, H5E_BADITER, FAIL, "open attribute count mismatch");

        
        for (u = 0; u < num_open_attr; u++) {
            unsigned long attr_fnum; 

            
            if (NULL == (*attr = (H5A_t *)H5VL_object_verify(attr_id_list[u], H5I_ATTR)))
                HGOTO_ERROR(H5E_ATTR, H5E_BADTYPE, FAIL, "not an attribute");

            
            if (H5F_get_fileno((*attr)->oloc.file, &attr_fnum) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_BADVALUE, FAIL, "can't get file serial number");

            
            if (!strcmp(name_to_open, (*attr)->shared->name) && loc->addr == (*attr)->oloc.addr &&
                loc_fnum == attr_fnum) {
                ret_value = true;
                break;
            } 
        }     
    }         

done:
    if (attr_id_list)
        H5MM_free(attr_id_list);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_update_shared(H5F_t *f, H5O_t *oh, H5A_t *attr, H5O_shared_t *update_sh_mesg)
{
    H5O_shared_t sh_mesg;             
    hsize_t      attr_rc;             
    htri_t       shared_mesg;         
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(attr);

    
    if (H5O_set_shared(&sh_mesg, &(attr->sh_loc)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, FAIL, "can't get shared message");

    
    if (H5O_msg_reset_share(H5O_ATTR_ID, attr) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTINIT, FAIL, "unable to reset attribute sharing");

    
    
    if ((shared_mesg = H5SM_try_share(f, oh, 0, H5O_ATTR_ID, attr, NULL)) == 0)
        HGOTO_ERROR(H5E_ATTR, H5E_BADMESG, FAIL, "attribute changed sharing status");
    else if (shared_mesg < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_BADMESG, FAIL, "can't share attribute");

    
    if (H5SM_get_refcount(f, H5O_ATTR_ID, &attr->sh_loc, &attr_rc) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't retrieve shared message ref count");

    
    if (attr_rc == 1)
        
        if (H5O__attr_link(f, oh, attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust attribute link count");

    
    if (H5SM_delete(f, oh, &sh_mesg) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to delete shared attribute in shared storage");

    
    if (update_sh_mesg)
        if (H5O_set_shared(update_sh_mesg, &(attr->sh_loc)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, FAIL, "can't get shared message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_write_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned H5_ATTR_UNUSED sequence,
                   void *_udata )
{
    H5O_iter_wrt_t    *udata       = (H5O_iter_wrt_t *)_udata; 
    H5O_chunk_proxy_t *chk_proxy   = NULL;                     
    bool               chk_dirtied = false;                    
    herr_t             ret_value   = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(mesg);
    assert(!udata->found);

    
    if (0 == strcmp(((H5A_t *)mesg->native)->shared->name, udata->attr->shared->name)) {
        
        if (NULL == (chk_proxy = H5O__chunk_protect(udata->f, oh, mesg->chunkno)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load object header chunk");

        
        if (((H5A_t *)mesg->native)->shared != udata->attr->shared) {
            
            assert(((H5A_t *)mesg->native)->shared->data);
            assert(udata->attr->shared->data);
            assert(((H5A_t *)mesg->native)->shared->data != udata->attr->shared->data);

            
            H5MM_memcpy(((H5A_t *)mesg->native)->shared->data, udata->attr->shared->data,
                        udata->attr->shared->data_size);
        } 

        
        mesg->dirty = true;
        chk_dirtied = true;

        
        if (H5O__chunk_unprotect(udata->f, chk_proxy, chk_dirtied) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, H5_ITER_ERROR,
                        "unable to unprotect object header chunk");
        chk_proxy = NULL;

        
        if (mesg->flags & H5O_MSG_FLAG_SHARED)
            if (H5O__attr_update_shared(udata->f, oh, udata->attr, (H5O_shared_t *)mesg->native) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, H5_ITER_ERROR,
                            "unable to update attribute in shared storage");

        
        oh->mesgs_modified = H5O_MODIFY;

        
        udata->found = true;

        
        ret_value = H5_ITER_STOP;
    } 

done:
    
    if (chk_proxy && H5O__chunk_unprotect(udata->f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_write(const H5O_loc_t *loc, H5A_t *attr)
{
    H5O_t      *oh = NULL;           
    H5O_ainfo_t ainfo;               
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(attr);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (H5A__dense_write(loc->file, &ainfo, attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "error updating attribute");
    } 
    else {
        H5O_iter_wrt_t      udata; 
        H5O_mesg_operator_t op;    

        
        udata.f     = loc->file;
        udata.attr  = attr;
        udata.found = false;

        
        op.op_type  = H5O_MESG_OP_LIB;
        op.u.lib_op = H5O__attr_write_cb;
        if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "error updating attribute");

        
        if (!udata.found)
            HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't locate open attribute?");
    } 

    
    if (H5O_touch_oh(loc->file, oh, false) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_rename_chk_cb(H5O_t H5_ATTR_UNUSED *oh, H5O_mesg_t *mesg ,
                        unsigned H5_ATTR_UNUSED sequence, void *_udata )
{
    H5O_iter_ren_t *udata     = (H5O_iter_ren_t *)_udata; 
    herr_t          ret_value = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(oh);
    assert(mesg);
    assert(!udata->found);

    
    if (strcmp(((H5A_t *)mesg->native)->shared->name, udata->new_name) == 0) {
        
        udata->found = true;

        
        ret_value = H5_ITER_STOP;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_rename_mod_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned H5_ATTR_UNUSED sequence,
                        void *_udata )
{
    H5O_iter_ren_t    *udata       = (H5O_iter_ren_t *)_udata; 
    H5O_chunk_proxy_t *chk_proxy   = NULL;                     
    bool               chk_dirtied = false;                    
    herr_t             ret_value   = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(mesg);
    assert(!udata->found);

    
    if (strcmp(((H5A_t *)mesg->native)->shared->name, udata->old_name) == 0) {
        unsigned old_version = ((H5A_t *)mesg->native)->shared->version; 

        
        if (NULL == (chk_proxy = H5O__chunk_protect(udata->f, oh, mesg->chunkno)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load object header chunk");

        
        H5MM_xfree(((H5A_t *)mesg->native)->shared->name);
        ((H5A_t *)mesg->native)->shared->name = H5MM_xstrdup(udata->new_name);

        
        if (H5A__set_version(udata->f, ((H5A_t *)mesg->native)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTSET, H5_ITER_ERROR, "unable to update attribute version");

        
        mesg->dirty = true;
        chk_dirtied = true;

        
        if (H5O__chunk_unprotect(udata->f, chk_proxy, chk_dirtied) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, H5_ITER_ERROR,
                        "unable to unprotect object header chunk");
        chk_proxy = NULL;

        
        if (mesg->flags & H5O_MSG_FLAG_SHARED) {
            
            if (H5O__attr_update_shared(udata->f, oh, (H5A_t *)mesg->native, NULL) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, H5_ITER_ERROR,
                            "unable to update attribute in shared storage");
        } 
        else {
            
            assert(H5O_msg_is_shared(H5O_ATTR_ID, (H5A_t *)mesg->native) == false);

            
            if (strlen(udata->new_name) != strlen(udata->old_name) ||
                old_version != ((H5A_t *)mesg->native)->shared->version) {
                H5A_t *attr; 

                
                
                attr         = (H5A_t *)mesg->native;
                mesg->native = NULL;

                
                
                if (H5O__release_mesg(udata->f, oh, mesg, false) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, H5_ITER_ERROR,
                                "unable to release previous attribute");

                oh->mesgs_modified = H5O_MODIFY_CONDENSE;

                
                
                if (H5O__msg_append_real(udata->f, oh, H5O_MSG_ATTR, (mesg->flags | H5O_MSG_FLAG_DONTSHARE),
                                         0, attr) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, H5_ITER_ERROR,
                                "unable to relocate renamed attribute in header");

                
                assert(H5O_msg_is_shared(H5O_ATTR_ID, attr) == false);

                
                H5A__close(attr);
            } 
        }     

        
        oh->mesgs_modified |= H5O_MODIFY;

        
        udata->found = true;

        
        ret_value = H5_ITER_STOP;
    } 

done:
    
    if (chk_proxy && H5O__chunk_unprotect(udata->f, chk_proxy, chk_dirtied) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to unprotect object header chunk");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_rename(const H5O_loc_t *loc, const char *old_name, const char *new_name)
{
    H5O_t      *oh = NULL;           
    H5O_ainfo_t ainfo;               
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(loc->addr)

    
    assert(loc);
    assert(old_name);
    assert(new_name);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (H5A__dense_rename(loc->file, &ainfo, old_name, new_name) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "error updating attribute");
    } 
    else {
        H5O_iter_ren_t      udata; 
        H5O_mesg_operator_t op;    

        
        udata.f        = loc->file;
        udata.old_name = old_name;
        udata.new_name = new_name;
        udata.found    = false;

        
        op.op_type  = H5O_MESG_OP_LIB;
        op.u.lib_op = H5O__attr_rename_chk_cb;
        if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "error updating attribute");

        
        if (udata.found)
            HGOTO_ERROR(H5E_ATTR, H5E_EXISTS, FAIL, "attribute with new name already exists");

        
        op.op_type  = H5O_MESG_OP_LIB;
        op.u.lib_op = H5O__attr_rename_mod_cb;
        if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "error updating attribute");

        
        if (!udata.found)
            HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't locate attribute with old name");
    } 

    
    if (H5O_touch_oh(loc->file, oh, false) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O_attr_iterate_real(hid_t loc_id, const H5O_loc_t *loc, H5_index_t idx_type, H5_iter_order_t order,
                      hsize_t skip, hsize_t *last_attr, const H5A_attr_iter_op_t *attr_op, void *op_data)
{
    H5O_t           *oh = NULL;                
    H5O_ainfo_t      ainfo;                    
    H5A_attr_table_t atable    = {0, 0, NULL}; 
    herr_t           ret_value = FAIL;         

    FUNC_ENTER_NOAPI_NOINIT_TAG(loc->addr)

    
    assert(loc);
    assert(loc->file);
    assert(H5_addr_defined(loc->addr));
    assert(attr_op);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (skip > 0 && skip >= ainfo.nattrs)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index specified");

        
        if (H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
        oh = NULL;

        
        if ((ret_value = H5A__dense_iterate(loc->file, loc_id, &ainfo, idx_type, order, skip, last_attr,
                                            attr_op, op_data)) < 0)
            HERROR(H5E_ATTR, H5E_BADITER, "error iterating over attributes");
    } 
    else {
        
        if (H5A__compact_build_table(loc->file, oh, idx_type, order, &atable) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "error building attribute table");

        
        if (H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
        oh = NULL;

        
        if (skip > 0 && skip >= atable.num_attrs)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index specified");

        
        if ((ret_value = H5A__attr_iterate_table(&atable, skip, last_attr, loc_id, attr_op, op_data)) < 0)
            HERROR(H5E_ATTR, H5E_BADITER, "iteration operator failed");
    } 

done:
    
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");
    if (atable.attrs && H5A__attr_release_table(&atable) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to release attribute table");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__attr_iterate(hid_t loc_id, H5_index_t idx_type, H5_iter_order_t order, hsize_t skip, hsize_t *last_attr,
                  const H5A_attr_iter_op_t *attr_op, void *op_data)
{
    H5G_loc_t loc;              
    herr_t    ret_value = FAIL; 

    FUNC_ENTER_PACKAGE

    
    assert(attr_op);

    
    if (H5G_loc(loc_id, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a location");

    
    if ((ret_value =
             H5O_attr_iterate_real(loc_id, loc.oloc, idx_type, order, skip, last_attr, attr_op, op_data)) < 0)
        HERROR(H5E_ATTR, H5E_BADITER, "error iterating over attributes");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_remove_update(const H5O_loc_t *loc, H5O_t *oh, H5O_ainfo_t *ainfo)
{
    H5A_attr_table_t atable    = {0, 0, NULL}; 
    herr_t           ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    assert(loc);
    assert(oh);
    assert(ainfo);

    
    ainfo->nattrs--;

    
    if (H5_addr_defined(ainfo->fheap_addr) && ainfo->nattrs < oh->min_dense) {
        bool   can_convert = true; 
        size_t u;                  

        
        if (H5A__dense_build_table(loc->file, ainfo, H5_INDEX_NAME, H5_ITER_NATIVE, &atable) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "error building attribute table");

        
        for (u = 0; u < ainfo->nattrs; u++)
            if (H5O_msg_size_oh(loc->file, oh, H5O_ATTR_ID, (atable.attrs[u]), (size_t)0) >=
                H5O_MESG_MAX_SIZE) {
                can_convert = false;
                break;
            } 

        
        if (can_convert) {
            H5A_t *exist_attr      = NULL;
            htri_t found_open_attr = false;

            
            for (u = 0; u < ainfo->nattrs; u++) {
                htri_t shared_mesg; 

                
                if ((shared_mesg = H5O_msg_is_shared(H5O_ATTR_ID, (atable.attrs[u]))) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "error determining if message is shared");
                else if (shared_mesg == 0) {
                    
                    
                    if (H5O__attr_link(loc->file, oh, (atable.attrs[u])) < 0)
                        HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust attribute link count");
                } 
                else {
                    
                    (atable.attrs[u])->sh_loc.type = H5O_SHARE_TYPE_UNSHARED;
                } 

                
                
                if ((found_open_attr =
                         H5O__attr_find_opened_attr(loc, &exist_attr, (atable.attrs[u])->shared->name)) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "failed in finding opened attribute");

                
                if (found_open_attr && exist_attr) {
                    if (H5O__msg_append_real(loc->file, oh, H5O_MSG_ATTR, 0, 0, exist_attr) < 0)
                        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "can't create message");

                } 
                else if (H5O__msg_append_real(loc->file, oh, H5O_MSG_ATTR, 0, 0, (atable.attrs[u])) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "can't create message");
            } 

            
            if (H5A__dense_delete(loc->file, ainfo) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete dense attribute storage");
        } 
    }     

    
    
    if (H5O__msg_write_real(loc->file, oh, H5O_MSG_AINFO, H5O_MSG_FLAG_DONTSHARE, 0, ainfo) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update attribute info message");

    
    if (ainfo->nattrs == 0) {
        if (H5O__msg_remove_real(loc->file, oh, H5O_MSG_AINFO, H5O_ALL, NULL, NULL, true) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute info");
    } 

done:
    
    if (atable.attrs && H5A__attr_release_table(&atable) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to release attribute table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_remove_cb(H5O_t *oh, H5O_mesg_t *mesg , unsigned H5_ATTR_UNUSED sequence,
                    void *_udata )
{
    H5O_iter_rm_t *udata     = (H5O_iter_rm_t *)_udata; 
    herr_t         ret_value = H5_ITER_CONT;            

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(mesg);
    assert(!udata->found);

    
    if (strcmp(((H5A_t *)mesg->native)->shared->name, udata->name) == 0) {
        
        if (H5O__release_mesg(udata->f, oh, mesg, true) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTDELETE, H5_ITER_ERROR, "unable to convert into null message");

        
        oh->mesgs_modified = H5O_MODIFY_CONDENSE;

        
        udata->found = true;

        
        ret_value = H5_ITER_STOP;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_remove(const H5O_loc_t *loc, const char *name)
{
    H5O_t      *oh = NULL;              
    H5O_ainfo_t ainfo;                  
    htri_t      ainfo_exists = false;   
    herr_t      ret_value    = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(loc->addr)

    
    assert(loc);
    assert(name);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if ((ainfo_exists = H5A__get_ainfo(loc->file, oh, &ainfo)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (H5A__dense_remove(loc->file, &ainfo, name) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute in dense storage");
    } 
    else {
        H5O_iter_rm_t       udata; 
        H5O_mesg_operator_t op;    

        
        udata.f     = loc->file;
        udata.name  = name;
        udata.found = false;

        
        op.op_type  = H5O_MESG_OP_LIB;
        op.u.lib_op = H5O__attr_remove_cb;
        if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "error deleting attribute");

        
        if (!udata.found)
            HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't locate attribute");
    } 

    
    if (ainfo_exists)
        if (H5O__attr_remove_update(loc, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update attribute info");

    
    if (H5O_touch_oh(loc->file, oh, false) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__attr_remove_by_idx(const H5O_loc_t *loc, H5_index_t idx_type, H5_iter_order_t order, hsize_t n)
{
    H5O_t           *oh = NULL;                   
    H5O_ainfo_t      ainfo;                       
    htri_t           ainfo_exists = false;        
    H5A_attr_table_t atable       = {0, 0, NULL}; 
    herr_t           ret_value    = SUCCEED;      

    FUNC_ENTER_PACKAGE_TAG(loc->addr)

    
    assert(loc);

    
    if (NULL == (oh = H5O_pin(loc)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPIN, FAIL, "unable to pin object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if ((ainfo_exists = H5A__get_ainfo(loc->file, oh, &ainfo)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (H5A__dense_remove_by_idx(loc->file, &ainfo, idx_type, order, n) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute in dense storage");
    } 
    else {
        H5O_iter_rm_t       udata; 
        H5O_mesg_operator_t op;    

        
        if (H5A__compact_build_table(loc->file, oh, idx_type, order, &atable) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "error building attribute table");

        
        if (n >= atable.num_attrs)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index specified");

        
        udata.f     = loc->file;
        udata.name  = ((atable.attrs[n])->shared)->name;
        udata.found = false;

        
        op.op_type  = H5O_MESG_OP_LIB;
        op.u.lib_op = H5O__attr_remove_cb;
        if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "error deleting attribute");

        
        if (!udata.found)
            HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't locate attribute");
    } 

    
    if (ainfo_exists)
        if (H5O__attr_remove_update(loc, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update attribute info");

    
    if (H5O_touch_oh(loc->file, oh, false) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update time on object");

done:
    if (oh && H5O_unpin(oh) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPIN, FAIL, "unable to unpin object header");
    if (atable.attrs && H5A__attr_release_table(&atable) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to release attribute table");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__attr_count_real(H5F_t *f, H5O_t *oh, hsize_t *nattrs)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oh);
    assert(nattrs);

    
    if (oh->version > H5O_VERSION_1) {
        htri_t      ainfo_exists = false; 
        H5O_ainfo_t ainfo;                

        
        if ((ainfo_exists = H5A__get_ainfo(f, oh, &ainfo)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
        else if (ainfo_exists > 0)
            *nattrs = ainfo.nattrs;
        else
            *nattrs = 0;
    } 
    else {
        hsize_t  attr_count; 
        unsigned u;          

        
        attr_count = 0;
        for (u = 0; u < oh->nmesgs; u++)
            if (oh->mesg[u].type == H5O_MSG_ATTR)
                attr_count++;
        *nattrs = attr_count;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__attr_exists_cb(H5O_t H5_ATTR_UNUSED *oh, H5O_mesg_t *mesg , unsigned H5_ATTR_UNUSED sequence,
                    void *_udata )
{
    H5O_iter_xst_t *udata     = (H5O_iter_xst_t *)_udata; 
    herr_t          ret_value = H5_ITER_CONT;             

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(mesg);
    assert(udata->exists && !*udata->exists);

    
    if (strcmp(((H5A_t *)mesg->native)->shared->name, udata->name) == 0) {
        
        *udata->exists = true;

        
        ret_value = H5_ITER_STOP;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5O__attr_exists(const H5O_loc_t *loc, const char *name, bool *attr_exists)
{
    H5O_t      *oh = NULL;           
    H5O_ainfo_t ainfo;               
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(loc->addr)

    
    assert(loc);
    assert(name);
    assert(attr_exists);

    
    if (NULL == (oh = H5O_protect(loc, H5AC__READ_ONLY_FLAG, false)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTPROTECT, FAIL, "unable to load object header");

    
    ainfo.fheap_addr = HADDR_UNDEF;
    if (oh->version > H5O_VERSION_1) {
        
        if (H5A__get_ainfo(loc->file, oh, &ainfo) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
    } 

    
    if (H5_addr_defined(ainfo.fheap_addr)) {
        
        if (H5A__dense_exists(loc->file, &ainfo, name, attr_exists) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_BADITER, FAIL, "error checking for existence of attribute");
    } 
    else {
        H5O_iter_xst_t      udata; 
        H5O_mesg_operator_t op;    

        
        udata.name   = name;
        udata.exists = attr_exists;

        
        op.op_type  = H5O_MESG_OP_LIB;
        op.u.lib_op = H5O__attr_exists_cb;
        if (H5O__msg_iterate_real(loc->file, oh, H5O_MSG_ATTR, &op, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_BADITER, FAIL, "error checking for existence of attribute");
    } 

done:
    if (oh && H5O_unprotect(loc, oh, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTUNPROTECT, FAIL, "unable to release object header");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5O__attr_bh_info(H5F_t *f, H5O_t *oh, H5_ih_info_t *bh_info)
{
    H5HF_t *fheap      = NULL;    
    H5B2_t *bt2_name   = NULL;    
    H5B2_t *bt2_corder = NULL;    
    herr_t  ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(oh);
    assert(bh_info);

    
    if (oh->version > H5O_VERSION_1) {
        H5O_ainfo_t ainfo;                
        htri_t      ainfo_exists = false; 

        
        if ((ainfo_exists = H5A__get_ainfo(f, oh, &ainfo)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't check for attribute info message");
        else if (ainfo_exists > 0) {
            
            if (H5_addr_defined(ainfo.name_bt2_addr)) {
                
                if (NULL == (bt2_name = H5B2_open(f, ainfo.name_bt2_addr, NULL)))
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

                
                if (H5B2_size(bt2_name, &(bh_info->index_size)) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't retrieve B-tree storage info");
            } 

            
            if (H5_addr_defined(ainfo.corder_bt2_addr)) {
                
                if (NULL == (bt2_corder = H5B2_open(f, ainfo.corder_bt2_addr, NULL)))
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL,
                                "unable to open v2 B-tree for creation order index");

                
                if (H5B2_size(bt2_corder, &(bh_info->index_size)) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't retrieve B-tree storage info");
            } 

            
            if (H5_addr_defined(ainfo.fheap_addr)) {
                
                if (NULL == (fheap = H5HF_open(f, ainfo.fheap_addr)))
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

                
                if (H5HF_size(fheap, &(bh_info->heap_size)) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't retrieve B-tree storage info");
            } 
        }     
    }         

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for creation order index");

    FUNC_LEAVE_NOAPI(ret_value)
} 
