% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{plot_multi_fitted}
\alias{plot_multi_fitted}
\title{Plot Multiple Fitted Distributions}
\usage{
plot_multi_fitted(
  data,
  pdf_list,
  cdf_list,
  qf_list,
  params_list,
  dist_names = NULL,
  col_list = NULL,
  lty_list = NULL,
  lwd_list = NULL,
  main_pdf = "Fitted PDFs",
  main_cdf = "Fitted CDFs",
  main_qq = "QQ Plots",
  main_pp = "PP Plots",
  xlab = "x"
)
}
\arguments{
\item{data}{Numeric vector of observed data.}

\item{pdf_list}{List of PDF functions. Each function should take \code{x} and \code{par}.}

\item{cdf_list}{List of CDF functions. Each function should take \code{x} and \code{par}.}

\item{qf_list}{List of quantile functions (inverse CDF). Each function should take \code{p} and \code{par}.}

\item{params_list}{List of parameter vectors corresponding to each distribution.}

\item{dist_names}{Optional vector of distribution names.}

\item{col_list}{Optional vector of colors for each distribution.}

\item{lty_list}{Optional vector of line types for each distribution.}

\item{lwd_list}{Optional vector of line widths for each distribution.}

\item{main_pdf}{Title for PDF plot.}

\item{main_cdf}{Title for CDF plot.}

\item{main_qq}{Title for QQ plot.}

\item{main_pp}{Title for PP plot.}

\item{xlab}{Label for x-axis.}
}
\value{
NULL (plots are generated as a side effect)
}
\description{
Creates 2x2 plots for multiple fitted distributions:
Fitted PDFs, Fitted CDFs vs Empirical CDF, QQ-Plots, and PP-Plots.
}
\examples{
# Example Multiple Distributions
set.seed(1)
data <- rexp(200, 1.1)

# Exponential
pdf_exp <- function(x, par) par[1] * exp(-par[1] * x)
cdf_exp <- function(x, par) 1 - exp(-par[1] * x)
qf_exp  <- function(p, par) -log(1 - p) / par[1]

# Generalized Exponential
pdf_gexp <- function(x, par) {
  a <- par[1]; l <- par[2]
  a * l * exp(-l*x) * (1-exp(-l*x))^(a-1)
}
cdf_gexp <- function(x, par) {
  a <- par[1]; l <- par[2]
  (1-exp(-l*x))^a
}
qf_gexp <- function(p, par) {
  a <- par[1]; l <- par[2]
  -log(1 - p^(1/a)) / l
}

# Weibull
pdf_weibull <- function(x, par) {
  k <- par[1]; l <- par[2]
  (k/l) * (x/l)^(k-1) * exp(-(x/l)^k)
}
cdf_weibull <- function(x, par) {
  k <- par[1]; l <- par[2]
  1 - exp(-(x/l)^k)
}
qf_weibull <- function(p, par) {
  k <- par[1]; l <- par[2]
  l * (-log(1 - p))^(1/k)
}

# Normal
pdf_norm <- function(x, par) dnorm(x, par[1], par[2])
cdf_norm <- function(x, par) pnorm(x, par[1], par[2])
qf_norm <- function(p, par) qnorm(p, par[1], par[2])

data <- rexp(200, 1)
# Call the plot function
plot_multi_fitted(
  data = data,
  pdf_list = list(pdf_exp, pdf_gexp, pdf_weibull, pdf_norm),
  cdf_list = list(cdf_exp, cdf_gexp, cdf_weibull, cdf_norm),
  qf_list  = list(qf_exp, qf_gexp, qf_weibull, qf_norm),
  params_list = list(
    c(1.1),
    c(2, 1.3),
    c(1.5, 2),
    c(0, 1)
  ),
  dist_names = c("Exp", "GExp", "Weibull", "Normal"),
  col_list = c("blue", "red", "darkgreen", "purple"),
  lty_list = c(1, 2, 3, 4),
  lwd_list = c(3, 3, 3, 3)
)
}
