#' Find Best Split Using Correlation Criterion
#'
#' @description
#' Searches for the optimal split point that minimizes the average absolute
#' correlation between time series across child nodes.
#'
#' @param X_node Covariate matrix for the current node.
#' @param Y_node Time series matrix for the current node.
#' @param indices Original indices of the observations in the current node.
#' @param control The parameters used in split.
#' @return A list containing the best split found, including the statistic,
#'   split variable, split value, and indices for the left and right children.
#' @keywords internal
#' @importFrom stats cor
find_best_split_COR <- function(X_node, Y_node, indices, control) {
  n_node <- ncol(Y_node)
  p_features <- ncol(X_node)
  best_split <- list(stat = Inf, var_idx = NA, split_val = NA, left_indices = NA, right_indices = NA)
  minbucket = control$minbucket

  # Enforce a minimum bucket size relative to node size to prevent trivial splits
  current_minbucket <- minbucket
  current_minbucket <- max(minbucket, ceiling(n_node * 0.1))
  if(n_node < 2*current_minbucket){
    return(best_split)
  }

  cor_mat = abs(cor(Y_node))
  for (j in 1:p_features) {
    x_feature <- X_node[, j]
    x_sort = sort(X_node[, j])
    order_x = order(X_node[, j])
    cor_mat_j = cor_mat[order_x, order_x]
    val = rep(NA, n_node)
    for(i in current_minbucket:(n_node - current_minbucket)){
      if(i == current_minbucket){
        val[i] = sum(cor_mat_j[1:i, (i+1):n_node])
      }else{
        val[i] = val[i-1] + sum(cor_mat_j[i, (i+1):n_node]) - sum(cor_mat_j[1:(i-1), i])
      }
      current_stat = val[i]/(i*(n_node - i))
      if (current_stat < best_split$stat) {
        best_split$stat <- current_stat
        best_split$var_idx <- j
        c = x_sort[i]
        best_split$split_val <- c
        idx_L_node <- which(x_feature < c)
        idx_R_node <- which(x_feature >= c)
        best_split$left_indices <- indices[idx_L_node]
        best_split$right_indices <- indices[idx_R_node]
      }
    }
  }
  return(best_split)
}
