#' Lord-Wingersky Recursive Formula
#'
#' @description
#' Compute the raw score distribution for a given theta value using the
#' Lord-Wingersky recursive formula, given item-level probabilities of a
#' correct response.
#'
#' @param probs A numeric vector (or matrix) of probabilities that a given
#'   theta value will correctly answer each item. If a matrix is provided,
#'   it will be coerced to a numeric vector.
#'
#' @return A list with:
#' \describe{
#'   \item{x}{Vector of possible raw scores, from 0 to \code{ni}.}
#'   \item{probability}{Vector of probabilities for each raw score.}
#' }
#'
#' @export
lord_wingersky <- function(probs) {

  # coerce to numeric vector ---------------------------------------------------
  if (is.null(probs)) {
    stop("`probs` must not be NULL.")
  }

  if (is.matrix(probs) || is.data.frame(probs)) {
    probs <- as.numeric(probs)
  }

  if (!is.numeric(probs)) {
    stop("`probs` must be numeric.")
  }

  ni <- length(probs)
  if (ni < 1L) {
    stop("`probs` must have length at least 1 (one item).")
  }

  if (any(probs < 0 | probs > 1, na.rm = TRUE)) {
    stop("All elements of `probs` must be between 0 and 1.")
  }

  # Lord-Wingersky recursion ---------------------------------------------------
  # initial distribution after first item
  obs <- numeric(ni + 1L)  # raw scores 0..ni
  obs[1] <- 1 - probs[1]   # P(X = 0)
  obs[2] <- probs[1]       # P(X = 1)

  if (ni >= 2L) {
    for (i in 2:ni) {
      prev <- obs

      # score 0
      obs[1] <- prev[1] * (1 - probs[i])

      # scores 1..i-1
      if (i >= 2L) {
        for (r in 2:i) {
          obs[r] <- prev[r] * (1 - probs[i]) + prev[r - 1] * probs[i]
        }
      }

      # score i
      obs[i + 1L] <- prev[i] * probs[i]
    }
  }

  return(list(
    x = 0:ni,
    probability   = obs
  ))
}
