% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outcome_models.R
\name{outcome_count}
\alias{outcome_count}
\title{Simulate from count model given covariates}
\usage{
outcome_count(
  data,
  mean = NULL,
  par = NULL,
  outcome.name = "y",
  exposure = 1,
  remove = c("id", "num"),
  zero.inflation = NULL,
  overdispersion = NULL,
  ...
)
}
\arguments{
\item{data}{(data.table) Covariate data, usually the output of the covariate
model of a \link{Trial} object.}

\item{mean}{(formula, function) Either a formula specifying the design from
'data' or a function that maps \code{data} to the conditional mean value on the
link scale (see examples). If NULL all main-effects of the covariates will
be used, except columns that are defined via the \code{remove} argument.}

\item{par}{(numeric) Regression coefficients (default zero). Can be given as
a named list corresponding to the column names of \code{model.matrix}}

\item{outcome.name}{Name of outcome variable ("y")}

\item{exposure}{Exposure times. Either a scalar, vector or function.}

\item{remove}{Variables that will be removed from input \code{data} (if formula is
not specified).}

\item{zero.inflation}{vector of probabilities or a function of the covariates
'x' including an extra column 'rate' with the rate parameter.}

\item{overdispersion}{variance of gamma-frailty either given as a numeric
vector or a function of the covariates 'x' with an extra column 'rate'
holding the rate parameter 'rate'}

\item{...}{Additional arguments passed to \code{mean} and \code{exposure} function}
}
\value{
data.table
}
\description{
Simulate from count model with intensity \deqn{\lambda =
  \text{exposure-time}\exp(\text{par}^\top X)} where \eqn{X} is the design
matrix specified by the formula
}
\examples{
covariates <- function(n) data.frame(a = rbinom(n, 1, 0.5), x = rnorm(n))
trial <- Trial$new(covariates = covariates, outcome = outcome_count)
trial$args_model(
  mean = ~ 1 + a + x,
  par = c(2.5, 0.65, 0),
  overdispersion = 1 / 2,
  exposure = 2 # identical exposure time for all subjects
)
est <- function(data) {
  glm(y ~ a + x + offset(log(exposure)), data, family = poisson())
}
trial$simulate(1e4) |> est()

# intercept + coef for x default to 0 and regression coef for a takes
# the provided value
trial$simulate(1e4, par = c(a = 0.65)) |> est()
# trial$simulate(1e4, mean = ~ 1 + a, par = c("(Intercept)" = 1))

# define mean model that directly works on whole covariate data, incl id and
# num columns
trial$simulate(1e4, mean = \(x) with(x, exp(1 + 0.5 * a))) |>
  est()

# treatment-dependent exposure times
trial$simulate(1e4, exposure = function(dd) 1 - 0.5 * dd$a) |>
  head()
}
\seealso{
\link{outcome_binary} \link{outcome_continuous}
}
