% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/collapseHz.R
\name{collapseHz}
\alias{collapseHz}
\title{Collapse Horizons within Profiles Based on Pattern Matching}
\usage{
collapseHz(
  x,
  pattern = NULL,
  by = NULL,
  hzdesgn = hzdesgnname(x, required = TRUE),
  FUN = function(x, pattern, hzdesgn, ...) grepl(pattern, x[[hzdesgn]], ignore.case =
    FALSE),
  ...,
  AGGFUN = NULL,
  ignore_numerics = NULL,
  na.rm = FALSE
)
}
\arguments{
\item{x}{A \emph{SoilProfileCollection}}

\item{pattern}{\emph{character}. A regular expression pattern to match in \code{hzdesgn} column. Default:
\code{NULL}.}

\item{by}{\emph{character}. A column name specifying horizons that should be combined. Aggregation
will be applied to adjacent groups of layers within profiles that have the same value in \code{by}.
Used in lieu of \code{pattern} and \code{hzdesgn}. Default: \code{NULL}.}

\item{hzdesgn}{\emph{character}. Any character column containing horizon-level identifiers. Default:
\code{hzdesgnname(x, required = TRUE)}.}

\item{FUN}{\emph{function}. A function that returns a \emph{logical} vector equal in length to the number
of horizons in \code{x}. Used only when \code{pattern} is specified. See details.}

\item{...}{Additional arguments passed to the matching function \code{FUN}.}

\item{AGGFUN}{\emph{list}. A \emph{named} list containing custom aggregation functions. List element names
should match the column name that they transform. The functions defined should take three
arguments: \code{x} (a vector of horizon property values), \code{top} (a vector of top depths), and
\code{bottom} (a vector of bottom depths). Default: \code{NULL} applies \code{weighted.mean()} to all numeric
columns not listed in \code{ignore_numerics} and takes the dominant condition (value with greatest
aggregate thickness sum) for all other columns. See details.}

\item{ignore_numerics}{\emph{character}. Vector of column names that contain numeric values which
should \emph{not} be aggregated using \code{weighted.mean()}. For example, soil color "value" and
"chroma".}

\item{na.rm}{\emph{logical}. If \code{TRUE} \code{NA} values are ignored when calculating min/max boundaries for
each group and in weighted averages. If \code{FALSE} \code{NA} values are propagated to the result.
Default: \code{FALSE}.}
}
\value{
A \emph{SoilProfileCollection}
}
\description{
Combines layers and aggregates data by grouping adjacent horizons which match \code{pattern} in
\code{hzdesgn} or, alternately, share a common value in \code{by} argument. Numeric properties are combined
using the weighted average, and other properties are derived from the dominant condition based on
thickness of layers and values in each group.
}
\details{
If a custom matching function (\code{FUN}) is used, it should accept arbitrary additional arguments
via an ellipsis (\code{...}). It is not necessary to do anything with arguments, but the result should
match the number of horizons found in the input SoilProfileCollection \code{x}.

Custom aggregation functions defined in the \code{AGGFUN} argument should either return a single
vector value for each group*column combination, or should return a \emph{data.frame} object with named
columns. If the input column name is used as a column name in the result \emph{data.frame}, then the
values of that column name in the result \emph{SoilProfileCollection} will be replaced by the output
of the aggregation function. See examples.
}
\examples{
data(jacobs2000)

# calculate a new SPC with genhz column based on patterns
new_labels <- c("A", "E", "Bt", "Bh", "C")
patterns <- c("A", "E", "B.*t", "B.*h", "C")
jacobs2000_gen <- generalizeHz(jacobs2000, new = new_labels, pattern = patterns)

# use existing generalized horizon labels
i <- collapseHz(jacobs2000_gen, by = "genhz")

profile_id(i) <- paste0(profile_id(i), "_collapse")

plot(
  c(i, jacobs2000),
  color = "genhz",
  name = "name",
  name.style = "center-center",
  cex.names = 1
)
 
# custom pattern argument  
j <- collapseHz(jacobs2000,
                c(
                  `A` = "^A",
                  `E` = "E",
                  `Bt` = "[ABC]+t",
                  `C` = "^C",
                  `foo` = "bar"
                ))
profile_id(j) <- paste0(profile_id(j), "_collapse")
plot(c(j, jacobs2000), color = "clay")

# custom aggregation function for matrix_color_munsell
k <- collapseHz(jacobs2000,
                pattern = c(
                  `A` = "^A",
                  `E` = "E",
                  `Bt` = "[ABC]+t",
                  `C` = "^C",
                  `foo` = "bar"
                ),
                AGGFUN = list(
                  matrix_color_munsell = function(x, top, bottom) {
                    thk <- bottom - top
                    if (length(x) > 1) {
                      xord <- order(thk, decreasing = TRUE)
                      paste0(paste0(x[xord], " (t=", thk[xord], ")"), collapse = ", ")
                    } else
                      x
                  }
                )
              )
profile_id(k) <- paste0(profile_id(k), "_collapse_custom")

unique(k$matrix_color_munsell)

# custom aggregation function for matrix_color_munsell (returns data.frame)
m <- collapseHz(jacobs2000,
                pattern = c(
                  `A` = "^A",
                  `E` = "E",
                  `Bt` = "[ABC]+t",
                  `C` = "^C",
                  `foo` = "bar"
                ),
                AGGFUN = list(
                  matrix_color_munsell = function(x, top, bottom) {
                    thk <- bottom - top
                    if (length(x) > 1) {
                      xord <- order(thk, decreasing = TRUE)
                      data.frame(matrix_color_munsell = paste0(x, collapse = ";"),
                                 n_matrix_color = length(x))
                    } else {
                      data.frame(matrix_color_munsell = x, 
                                 n_matrix_color = length(x))
                    }
                  }
                )
              )
profile_id(m) <- paste0(profile_id(m), "_collapse_custom")

m$matrix_color_munsell.n_matrix_color
}
\seealso{
\code{hz_dissolve()}
}
\author{
Andrew G. Brown
}
