#' Simulate Pareto-type Dynamic GLM
#'
#' Generates a dynamic time-series where
#' y_t = 1 + Gamma(shape = 1, scale = 1/lambda_t),
#' and the inverse-scale lambda_t follows a stationary AR(1) process.
#'
#' @param N Integer > 1, series length.
#' @param q Integer >= 1, number of predictors (used only for interface compatibility;
#'           no covariates are currently used in the generator).
#'
#' @return List with components:
#' \item{Y}{Numeric vector of length N, Pareto-type observations (y >= 1).}
#' \item{lambda}{Numeric vector of length N, dynamic inverse-scale process.}
#' \item{G}{AR(1) persistence coefficient (|G| < 1).}
#' \item{sig2}{Innovation variance sigma^2.}
#'
#' @examples
#' set.seed(2)
#' dat <- simPareto(N = 500, q = 3)
#' plot(dat$lambda, type = "l")
#'
#' @export
simPareto <- function(N = 1000L, q = 4L) {
  # Input checks
  if (N <= 1) stop("N must be > 1")
  if (q < 1) stop("q must be >= 1")
  
  # AR(1) for inverse-scale lambda
  lambda <- numeric(N)
  lambda[1] <- stats::rgamma(1, shape = 2, rate = 2)  # stationary initial draw
  G <- stats::runif(1, min = 0.4, max = 0.8)           # stable AR coefficient
  sig2 <- 0.1                                   # innovation variance
  
  for (t in 2:N) {
    lambda[t] <- G * lambda[t - 1] + stats::rnorm(1, mean = 0, sd = sqrt(sig2))
    lambda[t] <- max(lambda[t], 1e-3)           # ensure positivity
  }
  
  # Pareto-type observation (y >= 1)
  Y <- 1 + stats::rgamma(N, shape = 1, rate = lambda)  # shape = 1, scale = 1/lambda
  
  list(Y = Y, lambda = lambda, G = G, sig2 = sig2)
}
