#' Node score in network dynamics
#'
#' This function evaluates the significance of one or more nodes in a network given a system.
#' It initially tests the dynamics of the network under a healthy state (i.e., without any perturbation).
#' Then, it removes the specified nodes and runs the ODE again.
#' The function compares the final state of the network under the perturbation to that of the healthy network to assess the importance of the nodes for network health.
#'
#' @param system A function defining the system's dynamics: `system(time, x, params)`, which returns a list of state deltas `dx`.
#' @param M The initial adjacency matrix of the network.
#' @param x0 Initial conditions of the network's nodes (numeric vector).
#' @param nodes Nodes of interest to be removed.
#' @param reduction A reduction function applied to the ODE solution. The output of this function must be a numeric value. The default is `mean`.
#' @param initial_params Either a list of initial parameters or a function of type `f(M) -> list` that takes the adjacency matrix of the network as input and returns the initial parameters of the system.
#' @param times A vector of time points for the ODE solver.
#'
#' @return A list `L` with the following entries:
#'
#' - \strong{L$score}: The score of the nodes.
#'
#' - \strong{L$normalized.score}: The normalized score of the nodes.
#'
#' @export
#' @examples
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'    
#'    
#'    # Get the raw score of a node in the network
#'    score <- Rato::node.raw.score(
#'        Rato::Michaelis.Menten  # Use the Michaelis-Menten equation
#'      , g$M               # The GRN as an adjacency matrix
#'      , g$initial_values  # The initial values of the genes
#'      , g$node_index("ko:K03171") # The gene of interest
#'      , initial_params = list('f' = 1, 'h'=2, 'B'=0.01) # Parameters of the Michaelis-Menten equation
#'    )
node.raw.score <- function(   system                  # Dynamics of the system
                            , M                       # Adjacency matrix of the graph
                            , x0                      # Initial value
                            , nodes                   # Node set whose score we want to check
                            , initial_params = list() # Either a list, or a function: M -> list()
                            , times = seq(0, 100, 1)  # Idk about this
                            , reduction = mean        # How to reduce dimensionality. Alternative is median or max.
                          ){
  # Sanity checks to verify that the input makes sense.

  assert(nrow(M) == ncol(M), "Error: The graph adjacency matrix needs to be a square matrix.")
  assert(length(x0) == nrow(M), "Error: The initial values column should have the same length as the adjacency matrix of the graph.")
  assert(is.list(initial_params) | is.function(initial_params), "Error: initial_params needs to be a List or a function: Matrix -> List. ")

  # Ok, everything seems fine.

  # Calculate the initial parameters for the healthy network
  if(is.list(initial_params)){
    params <- initial_params
    params$M <- M
  }
  else if(is.function(initial_params)){
    params <- initial_params(M);
    params$M <- M
  }

  # Calculate the trajectory of the healthy network
  healthy_trajectory <- deSolve::ode(func = system, y = x0, times=times, parms=params)

  # Initial state of the healthy trajectory
  xi <- reduction(healthy_trajectory[1, -1])
  # Final state of the healthy trajectory
  xf <- reduction(healthy_trajectory[dim(healthy_trajectory)[1], -1])

  # Remove all nodes of interest
  for(node in nodes){
    M[, node] <- 0
    M[node, ] <- 0
  }

  # Calculate the parameters for the unhealthy network
  if(is.list(initial_params)){
    params <- initial_params
    params$M <- M
  }
  else if(is.function(initial_params)){
    params <- initial_params(M);
    params$M <- M
  }

  # Calculate the trajectory for the unhealthy network
  unhealthy_trajectory <- deSolve::ode(func = system, y = x0, times=times, parms=params)

  # initial state of the unhealthy network
  xip <- reduction(unhealthy_trajectory[1, -1])
  # Healthy state of the unhealthy network
  xfp <- reduction(unhealthy_trajectory[dim(unhealthy_trajectory)[1], -1])

  # The score is the difference between the final state of the unhealthy network
  # and the final state of the healthy network
  score <- xf - xfp

  # The normalized score takes into account the initial state as well.
  nscore <- (xf/xi) - (xfp/xip)

  # Returns the list with the score and normalized score
  return(list("score"=score, "normalized.score" = nscore))
}

#' Node score in network dynamics with significance test
#'
#' This function evaluates the significance of one or more nodes in a network given a system.
#' It initially tests the dynamics of the network under a healthy state (i.e., without any perturbation).
#' Then, it removes the specified nodes and runs the ODE again.
#' The function compares the final state of the network under the perturbation to that of the healthy network to assess the importance of the nodes for network health.
#' Additionally, it uses bootstrap methods to estimate a p-value associated with this score.
#'
#' @param system A function defining the system's dynamics: `system(time, x, params)`, which returns a list of state deltas `dx`.
#' @param M The initial adjacency matrix of the network.
#' @param x0 Initial conditions of the network's nodes (numeric vector).
#' @param nodes Nodes of interest to be removed.
#' @param reduction A reduction function applied to the ODE solution. The output of this function must be a numeric value. The default is `mean`.
#' @param initial_params Either a list of initial parameters or a function of type `f(M) -> list` that takes the adjacency matrix of the network as input and returns the initial parameters of the system.
#' @param times A vector of time points for the ODE solver.
#' @param bootstrap_iterations The number of bootstrap iterations to run.
#' @param skip_equal_nodes Whether to check if the sampled nodes are equal to the input nodes in the bootstrap method. This should be `TRUE` when nodes contain a single element or when the network is small.
#'
#' @return A list `L` with the following entries:
#'
#' - \strong{L$score}: The score of the nodes.
#'
#' - \strong{L$normalized.score}: The normalized score of the nodes.
#'
#' - \strong{L$p.value}: A p-value associated with the significance of the score.
#'
#' - \strong{L$normalized.p.value}: A normalized p-value associated with the significance of the score.
#'
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'    
#'    
#'    # Get the raw score of a node in the network
#'    score <- Rato::node.score(
#'        Rato::Michaelis.Menten  # Use the Michaelis-Menten equation
#'      , g$M               # The GRN as an adjacency matrix
#'      , g$initial_values  # The initial values of the genes
#'      , g$node_index("ko:K03171") # The gene of interest
#'      , initial_params = list('f' = 1, 'h'=2, 'B'=0.01) # Parameters of the Michaelis-Menten equation
#'    )
#' }
node.score <- function(   system                  # Dynamics of the system
                          , M                       # Adjacency matrix of the graph
                          , x0                      # Initial value
                          , nodes                   # Node set whose score we want to check
                          , initial_params = list() # Either a list, or a function: M -> list()
                          , times = seq(0, 100, 1)  # Idk about this
                          , reduction = mean        # How to reduce dimensionality. Alternative is median or max.
                          , bootstrap_iterations = 100
                          , skip_equal_nodes = TRUE # If the in
){

  # Sanity checks to verify that the input makes sense.

  assert(nrow(M) == ncol(M), "Error: The graph adjacency matrix needs to be a square matrix.")
  assert(length(x0) == nrow(M), "Error: The initial values column should have the same length as the adjacency matrix of the graph.")
  assert(is.list(initial_params) | is.function(initial_params), "Error: initial_params needs to be a List or a function: Matrix -> List. ")
  assert((length(nodes) < nrow(M)), "Error: You can`t check the score of ALL nodes. This makes no sense.")

  # Ok, everything seems fine.

  n <- nrow(M) # Number of nodes
  k <- length(nodes) # Number of nodes of interest

  # Get the raw score
  raw <- node.raw.score(system, M, x0, nodes, initial_params, times, reduction)

  # Bootstrap to test for significance

  bootstrap_scores <- c()
  bootstrap_nscores <- c()

  for(iteration in 1:bootstrap_iterations){

    bootstrap_nodes <- sample(1:n, k, replace=FALSE)

    # If the bootstrap_nodes are equal to the nodes, we should skip them as it makes no sense
    # to include them in the bootstrap iterations.
    if(skip_equal_nodes) {
      if(identical(sort(nodes), sort(bootstrap_nodes))){
        # TODO: Implement this
        # We should sample again until this condition fails
      }
    }

    bootstrap_score <- node.raw.score(system, M, x0, bootstrap_nodes, initial_params, times, reduction)
    bootstrap_scores <- c(bootstrap_scores, bootstrap_score$score)
    bootstrap_nscores <- c(bootstrap_nscores, bootstrap_score$normalized.score)

  }

  p <- length(which(bootstrap_scores > raw$score)) / length(bootstrap_scores)
  np <- length(which(bootstrap_nscores > raw$normalized.score)) / length(bootstrap_scores)

  output <- raw
  output$p.value <- p
  output$normalized.p.value <- np

  return(output)
}

# TODO: Edge score
